<?php
/**
 * ============================================================================
 * buyer/settings.php - Buyer Settings Page
 * ============================================================================
 */
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/BuyerSystem.php';

if (!isLoggedIn() || getUserType() !== 'buyer') {
    header('Location: ' . APP_URL . '/auth/login.php');
    exit();
}

use App\Database;
$database = Database::getInstance();
$db = $database->getConnection();
$currentUserId = getUserId();
$buyerSystem = new BuyerSystem($database, $currentUserId);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $db->beginTransaction();
        
        // Update buyer profile
        $sql = "UPDATE buyer_profiles SET
                    business_name = ?,
                    business_phone = ?,
                    business_email = ?,
                    business_address = ?,
                    buying_capacity = ?,
                    updated_at = NOW()
                WHERE user_id = ?";
        $stmt = $db->prepare($sql);
        $stmt->execute([
            $_POST['business_name'],
            $_POST['business_phone'],
            $_POST['business_email'],
            $_POST['business_address'],
            $_POST['buying_capacity'],
            $currentUserId
        ]);
        
        // Update user info
        $sql = "UPDATE users SET
                    full_name = ?,
                    email = ?,
                    updated_at = NOW()
                WHERE id = ?";
        $stmt = $db->prepare($sql);
        $stmt->execute([
            $_POST['full_name'],
            $_POST['email'],
            $currentUserId
        ]);
        
        $db->commit();
        
        $_SESSION['flash_message'] = 'Settings updated successfully';
        $_SESSION['flash_type'] = 'success';
        
        header('Location: ' . APP_URL . '/buyer/settings.php');
        exit();
        
    } catch (Exception $e) {
        $db->rollBack();
        $error = $e->getMessage();
    }
}

$profile = $buyerSystem->getProfile();

$pageTitle = 'Buyer Settings';
$breadcrumbs = [['title' => 'Settings', 'url' => '']];

require_once __DIR__ . '/includes/base.php';
renderBuyerHeader($pageTitle, $breadcrumbs);
?>

<div class="row">
    <div class="col-lg-3">
        <div class="card shadow-sm">
            <div class="list-group list-group-flush">
                <a href="#profile" class="list-group-item list-group-item-action active" data-bs-toggle="pill">
                    <i class="bi bi-person me-2"></i>Profile
                </a>
                <a href="#business" class="list-group-item list-group-item-action" data-bs-toggle="pill">
                    <i class="bi bi-building me-2"></i>Business Info
                </a>
                <a href="#notifications" class="list-group-item list-group-item-action" data-bs-toggle="pill">
                    <i class="bi bi-bell me-2"></i>Notifications
                </a>
                <a href="#security" class="list-group-item list-group-item-action" data-bs-toggle="pill">
                    <i class="bi bi-shield me-2"></i>Security
                </a>
            </div>
        </div>
    </div>
    
    <div class="col-lg-9">
        <?php if (isset($error)): ?>
        <div class="alert alert-danger"><?php echo $error; ?></div>
        <?php endif; ?>
        
        <form method="POST">
            <div class="tab-content">
                <!-- Profile Tab -->
                <div class="tab-pane fade show active" id="profile">
                    <div class="card shadow-sm">
                        <div class="card-header bg-white py-3">
                            <h5 class="mb-0"><i class="bi bi-person me-2"></i>Personal Information</h5>
                        </div>
                        <div class="card-body">
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label class="form-label">Full Name</label>
                                    <input type="text" name="full_name" class="form-control" 
                                           value="<?php echo htmlspecialchars($profile['full_name']); ?>" required>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Phone Number</label>
                                    <input type="text" class="form-control" 
                                           value="<?php echo htmlspecialchars($profile['phone_number']); ?>" disabled>
                                    <small class="text-muted">Contact support to change phone number</small>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Email Address</label>
                                    <input type="email" name="email" class="form-control" 
                                           value="<?php echo htmlspecialchars($profile['email'] ?? ''); ?>">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Buyer Type</label>
                                    <input type="text" class="form-control" 
                                           value="<?php echo formatBuyerType($profile['buyer_type']); ?>" disabled>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Business Tab -->
                <div class="tab-pane fade" id="business">
                    <div class="card shadow-sm">
                        <div class="card-header bg-white py-3">
                            <h5 class="mb-0"><i class="bi bi-building me-2"></i>Business Information</h5>
                        </div>
                        <div class="card-body">
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label class="form-label">Business Name</label>
                                    <input type="text" name="business_name" class="form-control" 
                                           value="<?php echo htmlspecialchars($profile['business_name'] ?? ''); ?>">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Business Phone</label>
                                    <input type="text" name="business_phone" class="form-control" 
                                           value="<?php echo htmlspecialchars($profile['business_phone'] ?? ''); ?>">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Business Email</label>
                                    <input type="email" name="business_email" class="form-control" 
                                           value="<?php echo htmlspecialchars($profile['business_email'] ?? ''); ?>">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Buying Capacity</label>
                                    <select name="buying_capacity" class="form-select">
                                        <option value="Small" <?php echo ($profile['buying_capacity'] ?? '') === 'Small' ? 'selected' : ''; ?>>Small</option>
                                        <option value="Medium" <?php echo ($profile['buying_capacity'] ?? '') === 'Medium' ? 'selected' : ''; ?>>Medium</option>
                                        <option value="Large" <?php echo ($profile['buying_capacity'] ?? '') === 'Large' ? 'selected' : ''; ?>>Large</option>
                                    </select>
                                </div>
                                <div class="col-12">
                                    <label class="form-label">Business Address</label>
                                    <textarea name="business_address" class="form-control" rows="3"><?php echo htmlspecialchars($profile['business_address'] ?? ''); ?></textarea>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Notifications Tab -->
                <div class="tab-pane fade" id="notifications">
                    <div class="card shadow-sm">
                        <div class="card-header bg-white py-3">
                            <h5 class="mb-0"><i class="bi bi-bell me-2"></i>Notification Preferences</h5>
                        </div>
                        <div class="card-body">
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" id="emailNotif" checked>
                                <label class="form-check-label" for="emailNotif">
                                    <strong>Email Notifications</strong>
                                    <small class="d-block text-muted">Receive order updates via email</small>
                                </label>
                            </div>
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" id="smsNotif" checked>
                                <label class="form-check-label" for="smsNotif">
                                    <strong>SMS Notifications</strong>
                                    <small class="d-block text-muted">Receive important updates via SMS</small>
                                </label>
                            </div>
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" id="priceAlert" checked>
                                <label class="form-check-label" for="priceAlert">
                                    <strong>Price Alerts</strong>
                                    <small class="d-block text-muted">Get notified of price changes</small>
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Security Tab -->
                <div class="tab-pane fade" id="security">
                    <div class="card shadow-sm">
                        <div class="card-header bg-white py-3">
                            <h5 class="mb-0"><i class="bi bi-shield me-2"></i>Security</h5>
                        </div>
                        <div class="card-body">
                            <div class="mb-4">
                                <h6>Change Password</h6>
                                <a href="<?php echo APP_URL; ?>/profile/change-password.php" class="btn btn-outline-primary">
                                    <i class="bi bi-key me-2"></i>Change Password
                                </a>
                            </div>
                            <div>
                                <h6 class="text-danger">Danger Zone</h6>
                                <p class="text-muted">Permanently delete your buyer account</p>
                                <button type="button" class="btn btn-outline-danger" onclick="confirmDelete()">
                                    <i class="bi bi-trash me-2"></i>Delete Account
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="mt-3">
                <button type="submit" class="btn btn-primary btn-lg">
                    <i class="bi bi-save me-2"></i>Save Changes
                </button>
                <a href="<?php echo APP_URL; ?>/buyer/dashboard.php" class="btn btn-outline-secondary btn-lg">
                    Cancel
                </a>
            </div>
        </form>
    </div>
</div>

<script>
function confirmDelete() {
    Swal.fire({
        title: 'Delete Account?',
        text: 'This action cannot be undone. All your data will be permanently deleted.',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#ef4444',
        confirmButtonText: 'Yes, Delete My Account'
    }).then((result) => {
        if (result.isConfirmed) {
            showToast('Please contact support to delete your account', 'info');
        }
    });
}
</script>

<?php renderBuyerFooter(); ?>