<?php
/**
 * ============================================================================
 * learning/best-practices.php - Agricultural Best Practices Page
 * ============================================================================
 */
define('AIMS_ACCESS', true);
require_once __DIR__ . '/../config/config.php';

requireLogin();

$pageTitle = 'Best Practices';

// Best practices organized by category
$bestPractices = [
    'crop_management' => [
        'title' => 'Crop Management',
        'icon' => 'bi-flower3',
        'color' => 'success',
        'practices' => [
            [
                'title' => 'Crop Rotation',
                'description' => 'Rotate crops annually to prevent soil depletion and reduce pest buildup. Alternate between nitrogen-fixing crops (beans, groundnuts) and heavy feeders (maize, vegetables).',
                'benefits' => ['Improved soil health', 'Reduced pest pressure', 'Better yields']
            ],
            [
                'title' => 'Proper Spacing',
                'description' => 'Follow recommended plant spacing for each crop. Overcrowding leads to competition for nutrients, water, and sunlight.',
                'benefits' => ['Optimal growth', 'Better air circulation', 'Easier management']
            ],
            [
                'title' => 'Timely Planting',
                'description' => 'Plant at the right time based on rainfall patterns and crop requirements. Monitor weather forecasts and traditional indicators.',
                'benefits' => ['Higher germination rates', 'Better establishment', 'Optimal yields']
            ]
        ]
    ],
    'soil_health' => [
        'title' => 'Soil Health',
        'icon' => 'bi-droplet',
        'color' => 'primary',
        'practices' => [
            [
                'title' => 'Mulching',
                'description' => 'Cover soil with organic materials (grass, leaves, crop residues) to conserve moisture, suppress weeds, and add nutrients as they decompose.',
                'benefits' => ['Water conservation', 'Weed control', 'Soil enrichment']
            ],
            [
                'title' => 'Composting',
                'description' => 'Make compost from farm waste, kitchen scraps, and animal manure. Apply 2-3 tons per acre before planting.',
                'benefits' => ['Improved soil structure', 'Nutrient supply', 'Waste management']
            ],
            [
                'title' => 'Minimal Tillage',
                'description' => 'Reduce soil disturbance to preserve soil structure and beneficial organisms. Use conservation tillage methods when possible.',
                'benefits' => ['Soil structure preservation', 'Water retention', 'Reduced erosion']
            ]
        ]
    ],
    'water_management' => [
        'title' => 'Water Management',
        'icon' => 'bi-water',
        'color' => 'info',
        'practices' => [
            [
                'title' => 'Rainwater Harvesting',
                'description' => 'Collect and store rainwater in tanks, ponds, or underground reservoirs for use during dry spells.',
                'benefits' => ['Water security', 'Reduced costs', 'Drought resilience']
            ],
            [
                'title' => 'Drip Irrigation',
                'description' => 'Use drip irrigation systems to deliver water directly to plant roots, minimizing water waste and maximizing efficiency.',
                'benefits' => ['Water savings up to 50%', 'Reduced disease', 'Better yields']
            ],
            [
                'title' => 'Irrigation Scheduling',
                'description' => 'Water crops during critical growth stages (flowering, fruit development). Irrigate early morning or evening to reduce evaporation.',
                'benefits' => ['Optimal water use', 'Reduced stress', 'Better quality produce']
            ]
        ]
    ],
    'pest_disease' => [
        'title' => 'Pest & Disease Control',
        'icon' => 'bi-bug',
        'color' => 'warning',
        'practices' => [
            [
                'title' => 'Scout Regularly',
                'description' => 'Inspect crops at least twice weekly for signs of pests or diseases. Early detection enables timely intervention.',
                'benefits' => ['Early problem detection', 'Reduced losses', 'Lower control costs']
            ],
            [
                'title' => 'Use Resistant Varieties',
                'description' => 'Choose crop varieties that are resistant or tolerant to common local pests and diseases.',
                'benefits' => ['Reduced pesticide use', 'Lower costs', 'More reliable yields']
            ],
            [
                'title' => 'Integrated Pest Management',
                'description' => 'Combine cultural, biological, and chemical methods. Use pesticides only when necessary and at the right time.',
                'benefits' => ['Environmental safety', 'Cost effectiveness', 'Sustainable control']
            ]
        ]
    ],
    'harvesting_storage' => [
        'title' => 'Harvesting & Storage',
        'icon' => 'bi-basket',
        'color' => 'danger',
        'practices' => [
            [
                'title' => 'Harvest at Right Maturity',
                'description' => 'Harvest crops at optimal maturity for best quality and market value. Avoid harvesting too early or too late.',
                'benefits' => ['Better quality', 'Longer shelf life', 'Higher market value']
            ],
            [
                'title' => 'Proper Handling',
                'description' => 'Handle produce carefully to avoid bruising and damage. Use clean containers and avoid overfilling.',
                'benefits' => ['Reduced losses', 'Better appearance', 'Extended shelf life']
            ],
            [
                'title' => 'Clean Storage',
                'description' => 'Store produce in clean, well-ventilated structures. Keep away from pests and moisture. Use proper storage bags or containers.',
                'benefits' => ['Reduced post-harvest losses', 'Better quality retention', 'Longer storage life']
            ]
        ]
    ]
];

include_once __DIR__ . '/../includes/meta.php';
include_once __DIR__ . '/../includes/header.php';
?>

<div class="container-fluid py-4">
    <!-- Header -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h2 class="mb-1">
                        <i class="bi bi-star-fill text-warning"></i> Agricultural Best Practices
                    </h2>
                    <p class="text-muted mb-0">Proven techniques to improve your farming success</p>
                </div>
                <a href="<?php echo APP_URL; ?>/learning/" class="btn btn-outline-primary">
                    <i class="bi bi-arrow-left"></i> Back to Learning Center
                </a>
            </div>
        </div>
    </div>

    <!-- Best Practices Categories -->
    <?php foreach ($bestPractices as $categoryKey => $category): ?>
        <div class="row mb-5">
            <div class="col-12">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-<?php echo $category['color']; ?> text-white py-3">
                        <h4 class="mb-0">
                            <i class="<?php echo $category['icon']; ?> me-2"></i>
                            <?php echo $category['title']; ?>
                        </h4>
                    </div>
                    <div class="card-body p-4">
                        <div class="row">
                            <?php foreach ($category['practices'] as $index => $practice): ?>
                                <div class="col-md-6 col-lg-4 mb-4">
                                    <div class="practice-card h-100">
                                        <div class="d-flex align-items-start mb-3">
                                            <div class="practice-number me-3">
                                                <?php echo $index + 1; ?>
                                            </div>
                                            <h5 class="mb-0"><?php echo $practice['title']; ?></h5>
                                        </div>
                                        
                                        <p class="text-muted mb-3">
                                            <?php echo $practice['description']; ?>
                                        </p>
                                        
                                        <div class="benefits-section">
                                            <h6 class="text-success mb-2">
                                                <i class="bi bi-check-circle-fill"></i> Benefits:
                                            </h6>
                                            <ul class="list-unstyled mb-0">
                                                <?php foreach ($practice['benefits'] as $benefit): ?>
                                                    <li class="mb-1">
                                                        <i class="bi bi-arrow-right-short text-success"></i>
                                                        <?php echo $benefit; ?>
                                                    </li>
                                                <?php endforeach; ?>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    <?php endforeach; ?>

    <!-- Quick Tips Section -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card border-0 shadow-sm bg-light">
                <div class="card-body p-4">
                    <h4 class="mb-4">
                        <i class="bi bi-lightbulb-fill text-warning"></i> Quick Tips for Success
                    </h4>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <div class="quick-tip">
                                <i class="bi bi-check-circle-fill text-success me-2"></i>
                                <strong>Keep farm records:</strong> Track all activities, inputs, and outputs
                            </div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <div class="quick-tip">
                                <i class="bi bi-check-circle-fill text-success me-2"></i>
                                <strong>Join farmer groups:</strong> Share knowledge and access better markets
                            </div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <div class="quick-tip">
                                <i class="bi bi-check-circle-fill text-success me-2"></i>
                                <strong>Continuous learning:</strong> Attend trainings and extension services
                            </div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <div class="quick-tip">
                                <i class="bi bi-check-circle-fill text-success me-2"></i>
                                <strong>Plan ahead:</strong> Prepare for planting season well in advance
                            </div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <div class="quick-tip">
                                <i class="bi bi-check-circle-fill text-success me-2"></i>
                                <strong>Diversify:</strong> Don't rely on a single crop or enterprise
                            </div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <div class="quick-tip">
                                <i class="bi bi-check-circle-fill text-success me-2"></i>
                                <strong>Market research:</strong> Know your market before planting
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Call to Action -->
    <div class="row">
        <div class="col-12">
            <div class="card border-0 shadow-sm text-center" style="background: linear-gradient(135deg, #059669 0%, #10b981 100%);">
                <div class="card-body p-5 text-white">
                    <h3 class="mb-3">Need More Help?</h3>
                    <p class="mb-4">Connect with agricultural extension officers or chat with other farmers</p>
                    <div class="d-flex justify-content-center gap-3">
                        <a href="<?php echo APP_URL; ?>/groups/" class="btn btn-light btn-lg">
                            <i class="bi bi-chat-dots"></i> Join Farmer Groups
                        </a>
                        <a href="<?php echo APP_URL; ?>/learning/" class="btn btn-outline-light btn-lg">
                            <i class="bi bi-book"></i> More Resources
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.practice-card {
    background: #f8f9fa;
    border-radius: 10px;
    padding: 20px;
    transition: all 0.3s ease;
}

.practice-card:hover {
    background: white;
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
    transform: translateY(-3px);
}

.practice-number {
    width: 40px;
    height: 40px;
    background: linear-gradient(135deg, #059669, #10b981);
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 1.2rem;
    flex-shrink: 0;
}

.benefits-section {
    background: white;
    border-left: 3px solid #22c55e;
    padding: 15px;
    border-radius: 5px;
}

.quick-tip {
    padding: 12px;
    background: white;
    border-radius: 8px;
    border-left: 3px solid #22c55e;
}
</style>

<?php
include_once __DIR__ . '/../includes/footer.php';
include_once __DIR__ . '/../includes/bottom-nav.php';
?>