# AIMS Platform - Object-Oriented Design

## Class Structure and Architecture

### Student: Musumba Jonathan (21/BCS/010/UMC)

### Institution: Metropolitan International University

---

## 1. CLASS DIAGRAM OVERVIEW

┌─────────────────────────────────────────────────────────────────────┐
│                    AIMS PLATFORM - CORE CLASSES                     │
└─────────────────────────────────────────────────────────────────────┘

                          ┌──────────────┐
                          │   Database   │
                          │              │
                          │ - host       │
                          │ - db_name    │
                          │ - username   │
                          │ - password   │
                          │ + conn       │
                          │              │
                          │+getConnection│
                          └──────┬───────┘
                                 │
                                 │ uses
                                 │
        ┌────────────────────────┼────────────────────────┐
        │                        │                        │
        ▼                        ▼                        ▼
┌──────────────┐        ┌──────────────┐        ┌──────────────┐
│     User     │        │   Product    │        │    Price     │
│              │        │              │        │              │
│ - id         │        │ - id         │        │ - id         │
│ - phone      │        │ - seller_id  │        │ - product    │
│ - password   │        │ - name       │        │ - location   │
│ - user_type  │        │ - category   │        │ - price      │
│              │        │ - quantity   │        │ - date       │
│+register()   │        │ - price      │        │              │
│+login()      │        │              │        │+getRecent()  │
│+verify()     │        │+create()     │        │+compare()    │
│+update()     │        │+update()     │        │+getPrediction│
└──────┬───────┘        │+delete()     │        └──────────────┘
       │                │+search()     │
       │                └──────────────┘
       │
       │ inherits
       │
       ├──────────────┬──────────────┬──────────────┐
       │              │              │              │
       ▼              ▼              ▼              ▼
┌──────────┐   ┌──────────┐   ┌──────────┐   ┌──────────┐
│  Farmer  │   │  Buyer   │   │Extension │   │  Admin   │
│          │   │          │   │ Officer  │   │          │
│+addFarm()│   │+placeOrder  │+schedule()│   │+approve()│
│+addProduct  │+trackOrder  │+report()  │   │+moderate()│
│+trackFarm   │+rateSuppl   │+assist()  │   │+configure│
└──────────┘   └──────────┘   └──────────┘   └──────────┘

---

## 2. BASE CLASSES

### 2.1 Database Class

/**

* Database Connection Handler
* Singleton pattern for database connections
 */
class Database {
    // Properties
    private $host = DB_HOST;
    private $db_name = DB_NAME;
    private $username = DB_USER;
    private $password = DB_PASS;
    public $conn;

    // Methods
    public function getConnection(): PDO
    public function closeConnection(): void
    public function beginTransaction(): bool
    public function commit(): bool
    public function rollback(): bool

    // Static
    private static $instance = null;
    public static function getInstance(): Database
}

---

### 2.2 BaseModel Class (Abstract)

php
/**

* Base Model - Parent for all models
* Provides common CRUD operations
 */
abstract class BaseModel {
    // Properties
    protected $db;
    protected $table;
    protected $primaryKey = 'id';

    // Constructor
    public function __construct(PDO $db)

    // Abstract methods (must be implemented)
    abstract protected function validate(): array

    // Common methods
    public function find(int $id): ?array
    public function findAll(array $conditions = []): array
    public function create(array $data): ?int
    public function update(int $id, array $data): bool
    public function delete(int $id): bool
    public function exists(int $id): bool
    public function count(array $conditions = []): int

    // Helper methods
    protected function sanitize($data)
    protected function executeQuery(string $query, array $params = []): PDOStatement
}

---

## 3. USER MANAGEMENT CLASSES

### 3.1 User Class

`php
/**

* User Model - Core user management
 */
class User extends BaseModel {
    // Properties
    protected $table = 'users';
    private $id;
    private $phone_number;
    private $password_hash;
    private $full_name;
    private $email;
    private $user_type;
    private $region_id;
    private $district_id;
    private $subcounty_id;
    private $parish_id;
    private $village_id;
    private $is_verified;
    private $is_active;
    private $approval_status;

    // Authentication
    public function register(array $data): array
    public function login(string $phone, string $password): array
    public function verifyOTP(string $phone, string $otp): array
    public function resendOTP(string $phone): array
    public function logout(): bool
    public function resetPassword(string $phone): array
    public function changePassword(int $userId, string $oldPass, string $newPass): bool

    // Profile Management
    public function getProfile(int $userId): ?array
    public function updateProfile(int $userId, array $data): bool
    public function uploadProfilePhoto(int $userId, $file): ?string
    public function updateLocation(int $userId, array $location): bool

    // User Queries
    public function getUserByPhone(string $phone): ?array
    public function getUserById(int $id): ?array
    public function getUsersByType(string $type): array
    public function searchUsers(string $query, array $filters = []): array

    // Status Management
    public function activate(int $userId): bool
    public function deactivate(int $userId): bool
    public function verify(int $userId): bool
    public function updateLastLogin(int $userId): bool
    public function updateLastActivity(int $userId): bool

    // Validation
    protected function validate(): array
    private function validatePhone(string $phone): bool
    private function validatePassword(string $password): bool
    private function validateEmail(string $email): bool
}

---

### 3.2 Farmer Class (extends User)

`php
/**

 *Farmer - Specialized user type
 */
class Farmer extends User {
    // Additional properties
    private $farms = [];
    private $enterprises = [];
    private $products = [];

    // Farm Management
    public function addFarm(array $farmData): ?int
    public function getFarms(): array
    public function updateFarm(int $farmId, array $data): bool
    public function deleteFarm(int $farmId): bool
    
    // Enterprise Management
    public function addEnterprise(int $farmId, array $data): ?int
    public function getEnterprises(int $farmId = null): array
    public function updateEnterprise(int $enterpriseId, array $data): bool
    public function deleteEnterprise(int $enterpriseId): bool
    
    // Product Management
    public function listProduct(array $productData): ?int
    public function getMyProducts(): array
    public function updateProduct(int $productId, array $data): bool
    public function deleteProduct(int $productId): bool
    public function markAsSold(int $productId): bool
    
    // Financial Tracking
    public function addTransaction(array $data): ?int
    public function getTransactions(array $filters = []): array
    public function getFinancialSummary(string $period): array
    public function getIncomeExpenseReport(int $year, int $month = null): array
    
    // Reminders
    public function setReminder(array $data): ?int
    public function getReminders(string $status = 'pending'): array
    public function completeReminder(int $reminderId): bool
}

### 3.3 Buyer Class (extends User)

`php
/**

 *Buyer - Procurement and purchasing
 */
class Buyer extends User {
    // Buyer-specific properties
    private $buyer_type;
    private $business_name;
    private $business_registration;
    private $tax_id;

    // Order Management
    public function placeOrder(int $productId, array $orderData): ?int
    public function getOrders(array $filters = []): array
    public function getOrderDetails(int $orderId): ?array
    public function cancelOrder(int $orderId): bool
    public function updateOrderStatus(int $orderId, string $status): bool
    
    // Supplier Management
    public function rateSupplier(int $supplierId, int $orderId, array $ratings): ?int
    public function getSupplierRating(int $supplierId): ?float
    public function getSuppliers(): array
    public function getFavoriteSuppliers(): array
    
    // Bulk Orders
    public function createBulkOrder(array $data): ?int
    public function getBulkOrders(): array
    public function reviewBids(int $bulkOrderId): array
    public function acceptBid(int $bidId): bool
    public function rejectBid(int $bidId): bool
    
    // Contracts
    public function initiateContract(int $farmerId, array $contractData): ?int
    public function getContracts(): array
    public function monitorContract(int $contractId): array
    public function recordDelivery(int $contractId, array $deliveryData): ?int
    
    // Preferences & Alerts
    public function setPreferences(array $preferences): bool
    public function getPreferences(): array
    public function addToWatchlist(string $productType, float $targetPrice): ?int
    public function getWatchlist(): array
}

---

### 3.4 ExtensionOfficer Class (extends User)

`php
/**
 *
 *Extension Officer - Agricultural extension services
 */
class ExtensionOfficer extends User {
    // Properties
    private $employee_id;
    private $organization;
    private $qualification;
    private $specialization;
    private $assigned_area;

    // Service Management
    public function scheduleService(array $serviceData): ?int
    public function getScheduledServices(array $filters = []): array
    public function updateService(int $serviceId, array $data): bool
    public function cancelService(int $serviceId): bool
    public function completeService(int $serviceId): bool
    
    // Participant Management
    public function getParticipants(int $serviceId): array
    public function markAttendance(int $serviceId, array $attendance): bool
    public function collectFeedback(int $serviceId, int $userId, array $feedback): ?int
    
    // Farmer Support
    public function respondToQuery(int $queryId, string $response): bool
    public function provideConsultation(int $farmerId, array $data): ?int
    public function visitFarm(int $farmId, array $visitData): ?int
    
    // Disease Management
    public function verifyDiseaseReport(int $reportId, array $verification): bool
    public function recommendTreatment(int $diseaseId): array
    
    // Content Creation
    public function createTip(array $tipData): ?int
    public function uploadResource(array $resourceData): ?int
    public function updateKnowledgeBase(int $contentId, array $data): bool
    
    // Reporting
    public function submitReport(array $reportData): ?int
    public function getReports(array $filters = []): array
    public function logActivity(array $activityData): ?int
    public function getActivityLog(): array
    public function getPerformanceMetrics(): array
    
    // Assignment
    public function getAssignment(): ?array
    public function updateCoverageArea(array $area): bool
    public function requestSupport(string $message): bool
}

---

### 3.5 Admin Class (extends User)

`php
/**
 *System Administrator
 */
class Admin extends User {
    // Properties
    private $role;
    private $permissions = [];

    // User Management
    public function getAllUsers(array $filters = []): array
    public function activateUser(int $userId): bool
    public function deactivateUser(int $userId): bool
    public function deleteUser(int $userId): bool
    public function resetUserPassword(int $userId): string
    
    // Approval System
    public function getPendingApprovals(): array
    public function approveRegistration(int $approvalId): bool
    public function rejectRegistration(int $approvalId, string $reason): bool
    public function reviewDocuments(int $approvalId): array
    
    // Role & Permission Management
    public function createRole(array $roleData): ?int
    public function assignPermissions(int $roleId, array $permissions): bool
    public function getUserPermissions(int $userId): array
    
    // Content Moderation
    public function moderateProducts(array $filters = []): array
    public function removeProduct(int $productId, string $reason): bool
    public function moderatePosts(): array
    public function removePost(int $postId, string $reason): bool
    public function handleReport(int $reportId, string $action): bool
    
    // Price Management
    public function addMarketPrice(array $priceData): ?int
    public function updateMarketPrice(int $priceId, array $data): bool
    public function importPrices(string $csvFile): bool
    public function verifyPriceReport(int $reportId): bool
    
    // Extension Management
    public function createExtensionAssignment(array $data): ?int
    public function assignSupervisor(int $assignmentId, int $supervisorId): bool
    public function reviewExtensionReport(int $reportId, array $review): bool
    public function getExtensionStatistics(): array
    
    // System Configuration
    public function updateSettings(array $settings): bool
    public function getSettings(): array
    public function configureAI(array $config): bool
    public function manageLocation(string $action, array $data): bool
    
    // Analytics & Reporting
    public function getDashboardStats(): array
    public function generateReport(string $type, array $params): array
    public function getAuditLogs(array $filters = []): array
    public function exportData(string $type, array $filters = []): string
    
    // System Maintenance
    public function backupDatabase(): bool
    public function checkSystemHealth(): array
    public function sendAnnouncement(array $data): bool
    public function broadcastMessage(string $message, array $recipients): bool
}

---

## 4. MARKETPLACE CLASSES

### 4.1 Product Class

`php
/**
 *Product Model - Marketplace listings
 */
class Product extends BaseModel {
    // Properties
    protected $table = 'products';
    private $id;
    private $seller_id;
    private $enterprise_id;
    private $product_name;
    private $category;
    private $description;
    private $quantity;
    private $unit;
    private $price_per_unit;
    private $location;
    private $images = [];
    private $status;
    private $views;

    // CRUD Operations
    public function create(array $data): ?int
    public function update(int $id, array $data): bool
    public function delete(int $id): bool
    public function get(int $id): ?array
    
    // Search & Filter
    public function search(string $query, array $filters = []): array
    public function getByCategory(string $category, array $filters = []): array
    public function getByLocation(string $location, array $filters = []): array
    public function getByPriceRange(float $min, float $max): array
    public function getAvailable(array $filters = []): array
    
    // Seller Operations
    public function getSellerProducts(int $sellerId): array
    public function updateStatus(int $id, string $status): bool
    public function incrementViews(int $id): bool
    
    // Media Management
    public function uploadImages(int $productId, array $files): array
    public function deleteImage(int $productId, string $imageUrl): bool
    public function getImages(int $productId): array
    
    // Related Operations
    public function getSimilar(int $productId, int $limit = 5): array
    public function getRecommended(int $userId, int $limit = 10): array
    
    // Validation
    protected function validate(): array
    private function validateCategory(string $category): bool
    private function validateQuantity(float $quantity, string $unit): bool
    private function validatePrice(float $price): bool
}

### 4.2 Order Class

`php
/**
 *Order Model - Purchase orders
 */
class Order extends BaseModel {
    // Properties
    protected $table = 'orders';
    private $id;
    private $order_number;
    private $product_id;
    private $buyer_id;
    private $seller_id;
    private $quantity;
    private $unit_price;
    private $total_amount;
    private $delivery_address;
    private $status;
    private $payment_status;
    private $payment_method;

    // Order Operations
    public function create(array $orderData): ?int
    public function update(int $orderId, array $data): bool
    public function cancel(int $orderId): bool
    public function get(int $orderId): ?array
    public function getByOrderNumber(string $orderNumber): ?array
    
    // Status Management
    public function updateStatus(int $orderId, string $status): bool
    public function updatePaymentStatus(int $orderId, string $status): bool
    public function confirmOrder(int $orderId): bool
    public function completeOrder(int $orderId): bool
    
    // Queries
    public function getBuyerOrders(int $buyerId, array $filters = []): array
    public function getSellerOrders(int $sellerId, array $filters = []): array
    public function getPendingOrders(): array
    public function getOrderHistory(int $userId, string $userType): array
    
    // Statistics
    public function getOrderStats(int $userId, string $userType): array
    public function getTotalSales(int $sellerId, string $period): float
    public function getTotalPurchases(int $buyerId, string $period): float
    
    // Validation
    protected function validate(): array
    private function generateOrderNumber(): string
    private function calculateTotal(float $quantity, float $unitPrice): float
}

### 4.3 Review Class

`php
/**
 *Review Model - Product/Seller reviews
 */
class Review extends BaseModel {
    // Properties
    protected $table = 'reviews';
    private $id;
    private $reviewer_id;
    private $reviewed_user_id;
    private $order_id;
    private $rating;
    private $review_text;

    // Review Operations
    public function create(array $reviewData): ?int
    public function update(int $reviewId, array $data): bool
    public function delete(int $reviewId): bool
    public function get(int $reviewId): ?array
    
    // Queries
    public function getUserReviews(int $userId): array
    public function getOrderReview(int $orderId): ?array
    public function getReviewsByRating(int $rating): array
    
    // Statistics
    public function getAverageRating(int $userId): ?float
    public function getRatingDistribution(int $userId): array
    public function getTotalReviews(int $userId): int
    
    // Validation
    protected function validate(): array
    private function validateRating(int $rating): bool
    private function canReview(int $reviewerId, int $orderId): bool
}

---

## 5. PRICE INTELLIGENCE CLASSES

### 5.1 Price Class

`php
/**
 *Market Price Model
 */
class Price extends BaseModel {
    // Properties
    protected $table = 'market_prices';
    private $id;
    private $product_type;
    private $market_location;
    private $price;
    private $unit;
    private $source;
    private $price_date;

    // Price Operations
    public function add(array $priceData): ?int
    public function update(int $id, array $data): bool
    public function get(int $id): ?array
    
    // Price Queries
    public function getCurrentPrices(string $productType, string $location = null): array
    public function getRecentPrices(string $productType, int $days = 7): array
    public function getLatestPrice(string $productType, string $location): ?array
    
    // Price Comparison
    public function comparePrices(string $productType, array $locations): array
    public function getPriceRange(string $productType): array
    public function getLowestPrice(string $productType): ?array
    public function getHighestPrice(string $productType): ?array
    
    // Historical Data
    public function getPriceHistory(string $productType, string $startDate, string $endDate): array
    public function getPriceTrends(string $productType, int $months = 12): array
    
    // Analytics
    public function calculateAverage(string $productType, int $days = 30): ?float
    public function getPriceVolatility(string $productType, int $period = 90): float
    public function getSeasonalPattern(string $productType): array
    
    // Predictions
    public function getPrediction(string $productType, int $daysAhead = 30): ?array
    
    // Validation
    protected function validate(): array
    private function validateProductType(string $type): bool
    private function validatePrice(float $price): bool
}

### 5.2 PriceAlert Class

`php
/**
 *Price Alert Model
 */
class PriceAlert extends BaseModel {
    // Properties
    protected $table = 'price_alerts';
    private $id;
    private $user_id;
    private $product_type;
    private $target_price;
    private $alert_condition;
    private $is_active;

    // Alert Operations
    public function create(array $alertData): ?int
    public function update(int $alertId, array $data): bool
    public function delete(int $alertId): bool
    public function activate(int $alertId): bool
    public function deactivate(int $alertId): bool
    
    // Alert Queries
    public function getUserAlerts(int $userId): array
    public function getActiveAlerts(): array
    public function getTriggeredAlerts(): array
    
    // Alert Processing
    public function checkAlerts(): array
    public function processAlert(int $alertId, float $currentPrice): bool
    public function sendNotification(int $alertId, float $price): bool
    
    // Validation
    protected function validate(): array
    private function validateCondition(string $condition, float $targetPrice, float $currentPrice): bool
}

---

## 6. COMMUNICATION CLASSES

### 6.1 Chat Class

`php
/**
 *Chat Model - Chat rooms
 */
class Chat extends BaseModel {
    // Properties
    protected $table = 'chats';
    private $id;
    private $chat_type;
    private $chat_name;
    private $chat_avatar;
    private $created_by;

    // Chat Operations
    public function create(array $chatData): ?int
    public function update(int $chatId, array $data): bool
    public function delete(int $chatId): bool
    public function get(int $chatId): ?array
    
    // Member Management
    public function addMember(int $chatId, int $userId, string $role = 'member'): bool
    public function removeMember(int $chatId, int $userId): bool
    public function updateMemberRole(int $chatId, int $userId, string $role): bool
    public function getMembers(int $chatId): array
    public function isMember(int $chatId, int $userId): bool
    
    // Chat Queries
    public function getUserChats(int $userId): array
    public function getOneToOneChat(int $userId1, int $userId2): ?array
    public function getGroupChats(int $userId): array
    
    // Chat Management
    public function updateLastRead(int $chatId, int $userId): bool
    public function getUnreadCount(int $chatId, int $userId): int
    
    // Validation
    protected function validate(): array
    private function validateChatType(string $type): bool
}

### 6.2 Message Class

`php
/**
 *Message Model - Chat messages
 */
class Message extends BaseModel {
    // Properties
    protected $table = 'messages';
    private $id;
    private $chat_id;
    private $sender_id;
    private $message_type;
    private $content;
    private $media_url;
    private $is_system_message;
    private $encrypted_content;
    private $is_encrypted;

    // Message Operations
    public function send(array $messageData): ?int
    public function update(int $messageId, array $data): bool
    public function delete(int $messageId): bool
    public function get(int $messageId): ?array
    
    // Message Queries
    public function getChatMessages(int $chatId, int $limit = 50, int $offset = 0): array
    public function getRecentMessages(int $chatId, int $count = 20): array
    public function searchMessages(int $chatId, string $query): array
    
    // Media Management
    public function uploadMedia(int $messageId, $file): ?string
    public function getMediaMessages(int $chatId): array
    
    // Encryption
    public function encrypt(string $content, string $key): string
    public function decrypt(string $encrypted, string $key): string
    
    // Reactions
    public function addReaction(int $messageId, int $userId, string $reaction): bool
    public function removeReaction(int $messageId, int $userId): bool
    public function getReactions(int $messageId): array
    
    // Validation
    protected function validate(): array
    private function validateMessageType(string $type): bool
}

---

## 7. AI & LEARNING CLASSES

### 7.1 AIAssistant Class

`php
/**
 *AI Assistant - Conversational AI
 */
class AIAssistant {
    // Properties
    private $db;
    private $openai;
    private $knowledgeBase;
    private $sessionManager;

    // Constructor
    public function __construct(PDO $db)
    
    // Conversation
    public function processMessage(int $userId, string $message, string $language = 'en'): array
    public function generateResponse(string $message, array $context): string
    public function detectIntent(string $message): string
    
    // Context Management
    public function getSessionContext(string $sessionId): array
    public function updateSessionContext(string $sessionId, array $context): bool
    public function clearSession(string $sessionId): bool
    
    // Knowledge Base
    public function searchKnowledge(string $query): array
    public function addKnowledge(array $data): bool
    public function learnFromFeedback(int $conversationId, string $rating): bool
    
    // Specialized Functions
    public function provideFarmingAdvice(string $query, array $context): string
    public function diagnoseDiease(string $symptoms, ?string $imagePath = null): array
    public function getWeatherInfo(string $location): array
    public function getMarketPrices(string $productType): array
    
    // Translation
    public function translate(string $text, string $targetLanguage): string
    
    // Pattern Learning
    public function identifyPattern(string $message): ?array
    public function savePattern(array $pattern): bool
    public function updatePatternSuccess(int $patternId, bool $success): bool
    
    // Analytics
    public function getConversationStats(int $userId): array
    public function getSuccessRate(): float
    public function getPopularTopics(): array
}

### 7.2 AgriculturalTip Class

`php
/**
 *Agricultural Tips & Best Practices
 */
class AgriculturalTip extends BaseModel {
    // Properties
    protected $table = 'agricultural_tips';
    private $id;
    private $title;
    private $content;
    private $category;
    private $crop_type;
    private $season;
    private $language;
    private $media_url;
    private $created_by;
    private $is_published;

    // Tip Operations
    public function create(array $tipData): ?int
    public function update(int $tipId, array $data): bool
    public function delete(int $tipId): bool
    public function publish(int $tipId): bool
    public function unpublish(int $tipId): bool

    // Queries
    public function getByCategory(string $category): array
    public function getByCropType(string $cropType): array
    public function getBySeason(string $season): array
    public function getByLanguage(string $language): array
    public function search(string $query): array
    
    // User Interaction
    public function bookmark(int $userId, int $tipId): bool
    public function removeBookmark(int $userId, int $tipId): bool
    public function getUserBookmarks(int $userId): array
    
    // Validation
    protected function validate(): array
}

---

## 8. UTILITY CLASSES

### 8.1 Notification Class

`php
/**
 *Notification System
 */
class Notification extends BaseModel {
    // Properties
    protected $table = 'notifications';
    private $id;
    private $user_id;
    private $title;
    private $message;
    private $type;
    private $icon;
    private $link;
    private $is_read;

    // Notification Operations
    public function create(array $notificationData): ?int
    public function markAsRead(int $notificationId): bool
    public function markAllAsRead(int $userId): bool
    public function delete(int $notificationId): bool
    
    // Queries
    public function getUserNotifications(int $userId, bool $unreadOnly = false): array
    public function getUnreadCount(int $userId): int
    public function getRecent(int $userId, int $limit = 10): array
    
    // Push Notifications
    public function sendPush(int $userId, array $data): bool
    public function sendToMultiple(array $userIds, array $data): bool
    public function schedulePush(array $data, string $sendTime): bool
    
    // SMS Notifications
    public function sendSMS(string $phone, string $message): bool
    
    // Email Notifications
    public function sendEmail(string $email, string $subject, string $body): bool
    
    // Validation
    protected function validate(): array
}

---

### 8.2 FileUpload Class

`php
/**
 *File Upload Handler
 */
class FileUpload {
    // Properties
    private $allowedExtensions = [];
    private $maxFileSize;
    private $uploadPath;

    // Constructor
    public function __construct(array $allowedExtensions, int $maxFileSize)
    
    // Upload Operations
    public function upload($file, string $type = 'general'): ?string
    public function uploadMultiple(array $files, string $type = 'general'): array
    public function delete(string $filePath): bool
    
    // Image Operations
    public function uploadImage($file, array $dimensions = []): ?string
    public function resizeImage(string $imagePath, int $width, int $height): bool
    public function cropImage(string $imagePath, array $crop): bool
    
    // Validation
    public function validate($file): array
    private function validateExtension(string $extension): bool
    private function validateSize(int $size): bool
    private function validateImageDimensions(string $imagePath, array $dimensions): bool
    
    // Helper Methods
    private function generateFileName(string $originalName): string
    private function getUploadPath(string $type): string
    public function getFileUrl(string $filePath): string
}

---

### 8.3 SessionManager Class

`php
/**
 *
 *Session Management
 */
class SessionManager {
    // Session Operations
    public static function start(): void
    public static function destroy(): void
    public static function regenerate(): bool

    // Session Data
    public static function set(string $key, $value): void
    public static function get(string $key, $default = null)
    public static function has(string $key): bool
    public static function remove(string $key): void
    public static function clear(): void
    
    // Flash Messages
    public static function setFlash(string $key, $value): void
    public static function getFlash(string $key, $default = null)
    public static function hasFlash(string $key): bool
    
    // Security
    public static function isValid(): bool
    public static function updateActivity(): void
    public static function checkTimeout(): bool

}

---

### 8.4 Validator Class

`php
/**
 *Input Validation
 */
class Validator {
    // Validation Methods
    public static function required($value): bool
    public static function email(string $email): bool
    public static function phone(string $phone): bool
    public static function numeric($value): bool
    public static function min($value, int $min): bool
    public static function max($value, int $max): bool
    public static function length(string $value, int $min, int $max): bool
    public static function regex(string $value, string $pattern): bool

    // Date Validation
    public static function date(string $date, string $format = 'Y-m-d'): bool
    public static function dateAfter(string $date1, string $date2): bool
    public static function dateBefore(string $date1, string $date2): bool
    
    // File Validation
    public static function fileExtension(string $filename, array $allowed): bool
    public static function fileSize(int $size, int $maxSize): bool
    
    // Custom Validation
    public static function unique(PDO $db, string $table, string $column, $value, ?int $excludeId = null): bool
    public static function exists(PDO $db, string $table, string $column, $value): bool
}

---

## 9. CLASS RELATIONSHIPS

### Inheritance Hierarchy

BaseModel
    ├── User
    │   ├── Farmer
    │   ├── Buyer
    │   ├── ExtensionOfficer
    │   └── Admin
    ├── Product
    ├── Order
    ├── Price
    ├── Chat
    ├── Message
    └── Notification

### Composition

* User **has-a** Profile
* User **has-a** Preferences
* Farmer **has-many** Farms
* Farm **has-many** Enterprises
* Chat **has-many** Messages
* Chat **has-many** Members

### Association

*User **creates** Products
*Buyer **places** Orders
*User **sends** Messages
*User **receives** Notifications

---

## 10. DESIGN PATTERNS USED

1. **Singleton Pattern** - Database connection
2. **Factory Pattern** - User type creation
3. **Repository Pattern** - Data access layer
4. **Observer Pattern** - Notifications
5. **Strategy Pattern** - Payment methods
6. **Decorator Pattern** - User roles/permissions
7. **MVC Pattern** - Overall architecture

---
