<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../classes/Profile.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: /auth/login.php');
    exit();
}

use App\Profile;

// Get database connection using the singleton pattern
$database = \App\Database::getInstance();
$db = $database->getConnection();

$profile = new Profile($db);
$currentUserId = $_SESSION['user_id'];
$viewingUserId = isset($_GET['id']) ? (int)$_GET['id'] : $currentUserId;

$userData = $profile->getProfile($viewingUserId, $currentUserId);

if (!$userData) {
    header('Location: /dashboard/index.php');
    exit();
}

$isOwnProfile = ($viewingUserId == $currentUserId);
$isFriend = $userData['is_friend'] ?? false;
$stats = $profile->getProfileStats($viewingUserId);

$pageTitle = $userData['full_name'] . "'s Profile";
include_once __DIR__ . '/../includes/meta.php';
include_once __DIR__ . '/../includes/header.php';
?>

<!-- ═══════════════════════════════════════════════════════════════════════
     STYLES - Bootstrap-based profile page
     ═══════════════════════════════════════════════════════════════════════ -->
<style>
/* Base */
.profile-page {
    background: #f0f2f5;
    min-height: calc(100vh - 120px);
}

/* Loading Overlay */
.loading-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    display: none;
    align-items: center;
    justify-content: center;
    z-index: 9999;
}

.loading-overlay.show {
    display: flex;
}

/* Cover Photo */
.profile-cover {
    height: 400px;
    background-size: cover;
    background-position: center;
    background-color: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    position: relative;
    border-radius: 0 0 12px 12px;
    overflow: hidden;
}

.profile-cover-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: linear-gradient(to bottom, rgba(0,0,0,0.1), rgba(0,0,0,0.3));
}

.cover-edit-btn {
    position: absolute;
    bottom: 20px;
    right: 20px;
    background: rgba(255, 255, 255, 0.9);
    border: none;
    border-radius: 8px;
    padding: 10px 16px;
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s;
    z-index: 2;
}

.cover-edit-btn:hover {
    background: #fff;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

/* Profile Header */
.profile-header-card {
    background: #fff;
    border-radius: 12px;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    margin-top: -80px;
    margin-bottom: 20px;
    padding: 20px;
    position: relative;
    z-index: 1;
}

.profile-avatar-wrapper {
    position: relative;
    width: 168px;
    height: 168px;
    margin: 0 auto;
    margin-top: -84px;
}

.profile-avatar {
    width: 168px;
    height: 168px;
    border-radius: 50%;
    border: 5px solid #fff;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    object-fit: cover;
}

.avatar-edit-btn {
    position: absolute;
    bottom: 10px;
    right: 10px;
    background: #fff;
    border: 2px solid #e4e6eb;
    border-radius: 50%;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s;
}

.avatar-edit-btn:hover {
    background: #f0f2f5;
    transform: scale(1.1);
}

.profile-name {
    font-size: 32px;
    font-weight: 700;
    color: #1c1e21;
    margin: 16px 0 0;
    text-align: center;
}

.verified-badge {
    display: inline-flex;
    align-items: center;
    gap: 4px;
    background: #e7f3ff;
    color: #1877f2;
    padding: 4px 10px;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 600;
    margin-left: 8px;
}

.profile-tagline {
    text-align: center;
    font-size: 18px;
    color: #65676b;
    margin: 8px 0;
}

.profile-meta {
    display: flex;
    justify-content: center;
    gap: 20px;
    flex-wrap: wrap;
    margin: 12px 0;
    font-size: 15px;
    color: #65676b;
}

.profile-meta span {
    display: flex;
    align-items: center;
    gap: 6px;
}

.profile-actions {
    display: flex;
    justify-content: center;
    gap: 12px;
    margin-top: 20px;
    flex-wrap: wrap;
}

.btn-profile {
    padding: 10px 24px;
    border-radius: 8px;
    font-size: 15px;
    font-weight: 600;
    border: none;
    cursor: pointer;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    transition: all 0.2s;
}

.btn-primary-gradient {
    background: linear-gradient(135deg, #1877f2 0%, #0c5ca5 100%);
    color: #fff;
}

.btn-primary-gradient:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(24, 119, 242, 0.4);
    color: #fff;
}

.btn-outline-gray {
    background: #e4e6eb;
    color: #1c1e21;
}

.btn-outline-gray:hover {
    background: #d8dadf;
    color: #1c1e21;
}

/* Profile Completeness */
.profile-completeness {
    background: linear-gradient(135deg, #667eea1a 0%, #764ba21a 100%);
    border-left: 4px solid #667eea;
    border-radius: 12px;
    padding: 16px 20px;
    margin-bottom: 20px;
}

.completeness-bar {
    width: 100%;
    height: 8px;
    background: #e4e6eb;
    border-radius: 10px;
    overflow: hidden;
    margin-top: 10px;
}

.completeness-fill {
    height: 100%;
    background: linear-gradient(90deg, #667eea 0%, #764ba2 100%);
    border-radius: 10px;
    transition: width 0.3s ease;
}

/* Stats */
.profile-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 12px;
    margin-bottom: 20px;
}

.stat-card {
    background: #fff;
    border-radius: 12px;
    padding: 20px;
    text-align: center;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    transition: all 0.2s;
}

.stat-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 6px 20px rgba(0, 0, 0, 0.12);
}

.stat-value {
    font-size: 28px;
    font-weight: 700;
    color: #1c1e21;
    margin-bottom: 4px;
}

.stat-label {
    font-size: 14px;
    color: #65676b;
    font-weight: 500;
}

/* Profile Cards */
.profile-card {
    background: #fff;
    border-radius: 12px;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    padding: 20px;
    margin-bottom: 20px;
}

.card-header {
    font-size: 20px;
    font-weight: 700;
    color: #1c1e21;
    margin: 0 0 16px;
    padding-bottom: 12px;
    border-bottom: 2px solid #f0f2f5;
}

.section-subtitle {
    font-size: 16px;
    font-weight: 600;
    color: #65676b;
    margin: 0 0 12px;
}

.about-text {
    font-size: 15px;
    line-height: 1.6;
    color: #1c1e21;
}

/* Skills & Interests */
.skills-grid {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
}

.skill-badge {
    background: #e7f3ff;
    color: #1877f2;
    padding: 8px 14px;
    border-radius: 20px;
    font-size: 14px;
    font-weight: 600;
    display: inline-block;
}

.interest-badge {
    background: #e8f5e9;
    color: #43a047;
}

/* Experience & Education */
.experience-item {
    padding: 16px 0;
    border-bottom: 1px solid #f0f2f5;
}

.experience-item:last-child {
    border-bottom: none;
}

.exp-title {
    font-size: 17px;
    font-weight: 700;
    color: #1c1e21;
    margin-bottom: 4px;
}

.exp-company {
    font-size: 15px;
    color: #65676b;
    margin-bottom: 4px;
}

.exp-duration {
    font-size: 14px;
    color: #8a8d91;
    margin-bottom: 8px;
}

.exp-description {
    font-size: 14px;
    color: #1c1e21;
    line-height: 1.5;
    margin-top: 8px;
}

.credential-link {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    color: #1877f2;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    margin-top: 8px;
}

.credential-link:hover {
    text-decoration: underline;
}

/* Recommendations */
.recommendation-item {
    padding: 16px;
    background: #f7f8fa;
    border-radius: 10px;
    margin-bottom: 12px;
}

.recommender-info {
    display: flex;
    align-items: center;
    gap: 12px;
    margin-bottom: 12px;
}

.recommender-avatar {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    object-fit: cover;
}

.recommender-name {
    font-size: 15px;
    font-weight: 600;
    color: #1c1e21;
}

.recommender-relationship {
    font-size: 13px;
    color: #65676b;
}

.recommendation-text {
    font-size: 14px;
    color: #1c1e21;
    line-height: 1.6;
    font-style: italic;
}

/* Achievements */
.achievement-item {
    display: flex;
    gap: 12px;
    padding: 12px 0;
    border-bottom: 1px solid #f0f2f5;
}

.achievement-item:last-child {
    border-bottom: none;
}

.achievement-icon {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    background: linear-gradient(135deg, #ffd700 0%, #ffa500 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    color: #fff;
    flex-shrink: 0;
}

.achievement-info h4 {
    font-size: 15px;
    font-weight: 600;
    color: #1c1e21;
    margin: 0 0 4px;
}

.achievement-info p {
    font-size: 13px;
    color: #65676b;
    margin: 0;
}

/* Contact */
.contact-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 10px 0;
    border-bottom: 1px solid #f0f2f5;
}

.contact-item:last-of-type {
    border-bottom: none;
}

.contact-icon {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: #f0f2f5;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 18px;
    color: #1877f2;
}

.contact-item a {
    color: #1c1e21;
    text-decoration: none;
    font-size: 14px;
}

.contact-item a:hover {
    color: #1877f2;
}

.social-section {
    margin-top: 16px;
    padding-top: 16px;
    border-top: 1px solid #f0f2f5;
}

.social-section h4 {
    font-size: 14px;
    font-weight: 600;
    color: #65676b;
    margin-bottom: 12px;
}

.social-links {
    display: flex;
    gap: 10px;
}

.social-link {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: #f0f2f5;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #1877f2;
    text-decoration: none;
    font-size: 18px;
    transition: all 0.2s;
}

.social-link:hover {
    background: #1877f2;
    color: #fff;
    transform: translateY(-3px);
}

/* Photo Grid */
.photo-grid {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 8px;
}

.photo-item {
    aspect-ratio: 1;
    border-radius: 8px;
    overflow: hidden;
    cursor: pointer;
    transition: transform 0.2s;
}

.photo-item:hover {
    transform: scale(1.05);
}

.photo-item img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.see-all-link {
    display: block;
    text-align: center;
    color: #1877f2;
    font-weight: 600;
    font-size: 14px;
    text-decoration: none;
    padding: 12px 0 0;
    margin-top: 12px;
    border-top: 1px solid #f0f2f5;
}

.see-all-link:hover {
    text-decoration: underline;
}

/* Responsive */
@media (max-width: 768px) {
    .profile-cover {
        height: 200px;
    }
    
    .profile-avatar-wrapper {
        width: 120px;
        height: 120px;
        margin-top: -60px;
    }
    
    .profile-avatar {
        width: 120px;
        height: 120px;
    }
    
    .profile-name {
        font-size: 24px;
    }
    
    .profile-meta {
        font-size: 13px;
    }
    
    .profile-stats {
        grid-template-columns: repeat(2, 1fr);
    }
}
</style>

<!-- ═══════════════════════════════════════════════════════════════════════
     PAGE MARKUP
     ═══════════════════════════════════════════════════════════════════════ -->
<div class="profile-page">

    <!-- Loading Overlay -->
    <div class="loading-overlay" id="loadingOverlay">
        <div class="spinner-border text-light" style="width: 3rem; height: 3rem;" role="status">
            <span class="visually-hidden">Loading...</span>
        </div>
    </div>

    <!-- Cover Photo -->
    <div class="profile-cover" style="background-image: url('<?php echo $userData['cover_photo'] ? APP_URL . '/' . htmlspecialchars($userData['cover_photo']) : ''; ?>'); <?php echo !$userData['cover_photo'] ? 'background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);' : ''; ?>">
        <div class="profile-cover-overlay"></div>
        <?php if ($isOwnProfile): ?>
        <button class="cover-edit-btn" onclick="document.getElementById('coverPhotoInput').click();">
            <i class="bi bi-camera-fill me-2"></i>Edit Cover Photo
        </button>
        <input type="file" id="coverPhotoInput" accept="image/*" style="display: none;" onchange="uploadCoverPhoto(this)">
        <?php endif; ?>
    </div>

    <div class="container" style="max-width: 1200px;">
        
        <!-- Profile Header -->
        <div class="profile-header-card">
            <div class="profile-avatar-wrapper">
                <img src="<?php echo $userData['profile_picture'] ? APP_URL . '/' . htmlspecialchars($userData['profile_picture']) : APP_URL . '/assets/images/default-avatar.png'; ?>"
                     alt="Profile" class="profile-avatar">
                <?php if ($isOwnProfile): ?>
                <button class="avatar-edit-btn" onclick="document.getElementById('profilePictureInput').click();">
                    <i class="bi bi-camera-fill"></i>
                </button>
                <input type="file" id="profilePictureInput" accept="image/*" style="display: none;" onchange="uploadProfilePicture(this)">
                <?php endif; ?>
            </div>
            
            <h1 class="profile-name">
                <?php echo htmlspecialchars($userData['full_name']); ?>
                <?php if ($userData['is_verified']): ?>
                <span class="verified-badge">
                    <i class="bi bi-patch-check-fill"></i> Verified
                </span>
                <?php endif; ?>
            </h1>
            
            <?php if ($userData['tagline']): ?>
            <p class="profile-tagline"><?php echo htmlspecialchars($userData['tagline']); ?></p>
            <?php endif; ?>
            
            <div class="profile-meta">
                <?php if ($userData['visible_location']): ?>
                <span><i class="bi bi-geo-alt-fill"></i> <?php echo htmlspecialchars($userData['visible_location']); ?></span>
                <?php endif; ?>
                <span><i class="bi bi-calendar-check"></i> Joined <?php echo date('M Y', strtotime($userData['joined_date'])); ?></span>
                <?php if ($userData['occupation']): ?>
                <span><i class="bi bi-briefcase-fill"></i> <?php echo htmlspecialchars($userData['occupation']); ?></span>
                <?php endif; ?>
            </div>
            
            <div class="profile-actions">
                <?php if ($isOwnProfile): ?>
                    <a href="<?php echo APP_URL; ?>/profile/edit.php" class="btn-profile btn-primary-gradient">
                        <i class="bi bi-pencil-square"></i> Edit Profile
                    </a>
                    <a href="<?php echo APP_URL; ?>/profile/settings.php" class="btn-profile btn-outline-gray">
                        <i class="bi bi-gear-fill"></i> Settings
                    </a>
                <?php else: ?>
                    <?php if ($isFriend): ?>
                        <button class="btn-profile btn-outline-gray">
                            <i class="bi bi-person-check-fill"></i> Friends
                        </button>
                    <?php else: ?>
                        <button class="btn-profile btn-primary-gradient" onclick="sendFriendRequest(<?php echo $viewingUserId; ?>)">
                            <i class="bi bi-person-plus-fill"></i> Add Friend
                        </button>
                    <?php endif; ?>
                    <a href="<?php echo APP_URL; ?>/chat/index.php?user_id=<?php echo $viewingUserId; ?>" class="btn-profile btn-outline-gray">
                        <i class="bi bi-chat-dots-fill"></i> Message
                    </a>
                <?php endif; ?>
            </div>
        </div>

        <!-- Profile Completeness -->
        <?php if ($isOwnProfile && $stats['profile_completeness'] < 100): ?>
        <div class="profile-completeness">
            <div class="d-flex justify-content-between align-items-center mb-2">
                <span style="font-weight: 600; color: #333;">Complete your profile</span>
                <span style="font-weight: 700; color: #667eea;"><?php echo $stats['profile_completeness']; ?>%</span>
            </div>
            <div class="completeness-bar">
                <div class="completeness-fill" style="width: <?php echo $stats['profile_completeness']; ?>%;"></div>
            </div>
            <p style="font-size: 13px; color: #888; margin-top: 8px; margin-bottom: 0;">
                Add more information to help others know you better
            </p>
        </div>
        <?php endif; ?>

        <!-- Stats -->
        <div class="profile-stats">
            <div class="stat-card">
                <div class="stat-value">
                    <i class="bi bi-people-fill text-primary"></i> 
                    <?php echo number_format($stats['friends_count']); ?>
                </div>
                <div class="stat-label">Friends</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">
                    <i class="bi bi-box-seam text-success"></i> 
                    <?php echo number_format($stats['products_count']); ?>
                </div>
                <div class="stat-label">Products</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">
                    <i class="bi bi-star-fill text-warning"></i> 
                    <?php echo number_format($stats['avg_rating'], 1); ?>
                </div>
                <div class="stat-label">Rating (<?php echo $stats['reviews_count']; ?> reviews)</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">
                    <i class="bi bi-eye-fill text-info"></i> 
                    <?php echo number_format($stats['profile_views']); ?>
                </div>
                <div class="stat-label">Profile Views</div>
            </div>
        </div>

        <!-- Two Column Layout -->
        <div class="row">
            <!-- Main Content -->
            <div class="col-lg-8">
                
                <?php if ($userData['about_me'] || $userData['bio']): ?>
                <div class="profile-card">
                    <h3 class="card-header"><i class="bi bi-info-circle-fill text-primary me-2"></i>About</h3>
                    <?php if ($userData['bio']): ?>
                    <p style="font-weight: 600; color: #667eea; margin-bottom: 12px; font-size: 16px;">
                        <?php echo htmlspecialchars($userData['bio']); ?>
                    </p>
                    <?php endif; ?>
                    <?php if ($userData['about_me']): ?>
                    <p class="about-text"><?php echo nl2br(htmlspecialchars($userData['about_me'])); ?></p>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($userData['skills']) || !empty($userData['interests'])): ?>
                <div class="profile-card">
                    <h3 class="card-header"><i class="bi bi-lightbulb-fill text-warning me-2"></i>Skills & Interests</h3>
                    <?php if (!empty($userData['skills'])): ?>
                    <h4 class="section-subtitle">Skills</h4>
                    <div class="skills-grid mb-3">
                        <?php foreach ($userData['skills'] as $skill): ?>
                        <span class="skill-badge"><?php echo htmlspecialchars($skill); ?></span>
                        <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                    <?php if (!empty($userData['interests'])): ?>
                    <h4 class="section-subtitle">Interests</h4>
                    <div class="skills-grid">
                        <?php foreach ($userData['interests'] as $interest): ?>
                        <span class="skill-badge interest-badge"><?php echo htmlspecialchars($interest); ?></span>
                        <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($userData['experience'])): ?>
                <div class="profile-card">
                    <h3 class="card-header"><i class="bi bi-briefcase-fill text-success me-2"></i>Experience</h3>
                    <?php foreach ($userData['experience'] as $exp): ?>
                    <div class="experience-item">
                        <div class="exp-title"><?php echo htmlspecialchars($exp['title']); ?></div>
                        <div class="exp-company">
                            <?php echo htmlspecialchars($exp['organization']); ?>
                            <?php if ($exp['location']): ?>
                            · <?php echo htmlspecialchars($exp['location']); ?>
                            <?php endif; ?>
                        </div>
                        <div class="exp-duration">
                            <?php echo date('M Y', strtotime($exp['start_date'])) . ' - ' . ($exp['is_current'] ? 'Present' : date('M Y', strtotime($exp['end_date']))); ?>
                        </div>
                        <?php if ($exp['description']): ?>
                        <div class="exp-description"><?php echo nl2br(htmlspecialchars($exp['description'])); ?></div>
                        <?php endif; ?>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($userData['education'])): ?>
                <div class="profile-card">
                    <h3 class="card-header"><i class="bi bi-mortarboard-fill text-danger me-2"></i>Education</h3>
                    <?php foreach ($userData['education'] as $edu): ?>
                    <div class="experience-item">
                        <div class="exp-title"><?php echo htmlspecialchars($edu['institution']); ?></div>
                        <div class="exp-company">
                            <?php echo htmlspecialchars($edu['degree']); ?>
                            <?php if ($edu['field_of_study']): ?>
                            · <?php echo htmlspecialchars($edu['field_of_study']); ?>
                            <?php endif; ?>
                        </div>
                        <div class="exp-duration">
                            <?php echo $edu['start_year'] . ' - ' . $edu['end_year']; ?>
                            <?php if ($edu['grade']): ?>
                            · Grade: <?php echo htmlspecialchars($edu['grade']); ?>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($userData['certifications'])): ?>
                <div class="profile-card">
                    <h3 class="card-header"><i class="bi bi-award-fill text-info me-2"></i>Certifications & Licenses</h3>
                    <?php foreach ($userData['certifications'] as $cert): ?>
                    <div class="experience-item">
                        <div class="exp-title"><?php echo htmlspecialchars($cert['certification_name']); ?></div>
                        <div class="exp-company"><?php echo htmlspecialchars($cert['issuing_organization']); ?></div>
                        <div class="exp-duration">
                            Issued <?php echo date('M Y', strtotime($cert['issue_date'])); ?>
                            <?php if ($cert['expiry_date']): ?>
                            · Expires <?php echo date('M Y', strtotime($cert['expiry_date'])); ?>
                            <?php endif; ?>
                        </div>
                        <?php if ($cert['credential_url']): ?>
                        <a href="<?php echo htmlspecialchars($cert['credential_url']); ?>" target="_blank" class="credential-link">
                            <i class="bi bi-link-45deg"></i> View Credential
                        </a>
                        <?php endif; ?>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($userData['recommendations'])): ?>
                <div class="profile-card">
                    <h3 class="card-header"><i class="bi bi-chat-quote-fill text-secondary me-2"></i>Recommendations</h3>
                    <?php foreach ($userData['recommendations'] as $rec): ?>
                    <div class="recommendation-item">
                        <div class="recommender-info">
                            <img src="<?php echo $rec['recommender_photo'] ? '/' . htmlspecialchars($rec['recommender_photo']) : '/assets/images/default-avatar.png'; ?>" 
                                 alt="" class="recommender-avatar">
                            <div>
                                <div class="recommender-name"><?php echo htmlspecialchars($rec['recommender_name']); ?></div>
                                <div class="recommender-relationship"><?php echo htmlspecialchars($rec['relationship']); ?></div>
                            </div>
                        </div>
                        <div class="recommendation-text">"<?php echo nl2br(htmlspecialchars($rec['recommendation_text'])); ?>"</div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
            </div>
            
            <!-- Sidebar -->
            <div class="col-lg-4">
                
                <?php if (!empty($userData['achievements'])): ?>
                <div class="profile-card">
                    <h3 class="card-header"><i class="bi bi-trophy-fill text-warning me-2"></i>Achievements</h3>
                    <?php foreach (array_slice($userData['achievements'], 0, 5) as $achievement): ?>
                    <div class="achievement-item">
                        <div class="achievement-icon"><i class="bi bi-trophy-fill"></i></div>
                        <div class="achievement-info">
                            <h4><?php echo htmlspecialchars($achievement['achievement_name']); ?></h4>
                            <p><?php echo htmlspecialchars($achievement['achievement_description']); ?></p>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <div class="profile-card">
                    <h3 class="card-header"><i class="bi bi-telephone-fill text-success me-2"></i>Contact Information</h3>
                    <?php if ($userData['visible_phone']): ?>
                    <div class="contact-item">
                        <div class="contact-icon"><i class="bi bi-telephone-fill"></i></div>
                        <a href="tel:<?php echo htmlspecialchars($userData['visible_phone']); ?>">
                            <?php echo htmlspecialchars($userData['visible_phone']); ?>
                        </a>
                    </div>
                    <?php endif; ?>
                    <?php if ($userData['whatsapp_number']): ?>
                    <div class="contact-item">
                        <div class="contact-icon"><i class="bi bi-whatsapp"></i></div>
                        <a href="https://wa.me/<?php echo htmlspecialchars($userData['whatsapp_number']); ?>" target="_blank">WhatsApp</a>
                    </div>
                    <?php endif; ?>
                    <?php if ($userData['visible_email']): ?>
                    <div class="contact-item">
                        <div class="contact-icon"><i class="bi bi-envelope-fill"></i></div>
                        <a href="mailto:<?php echo htmlspecialchars($userData['visible_email']); ?>">
                            <?php echo htmlspecialchars($userData['visible_email']); ?>
                        </a>
                    </div>
                    <?php endif; ?>
                    <?php if ($userData['website']): ?>
                    <div class="contact-item">
                        <div class="contact-icon"><i class="bi bi-globe"></i></div>
                        <a href="<?php echo htmlspecialchars($userData['website']); ?>" target="_blank">Website</a>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($userData['facebook_url'] || $userData['twitter_url'] || $userData['instagram_url'] || $userData['linkedin_url']): ?>
                    <div class="social-section">
                        <h4>Social Media</h4>
                        <div class="social-links">
                            <?php if ($userData['facebook_url']): ?>
                            <a href="<?php echo htmlspecialchars($userData['facebook_url']); ?>" class="social-link" target="_blank">
                                <i class="bi bi-facebook"></i>
                            </a>
                            <?php endif; ?>
                            <?php if ($userData['twitter_url']): ?>
                            <a href="<?php echo htmlspecialchars($userData['twitter_url']); ?>" class="social-link" target="_blank">
                                <i class="bi bi-twitter"></i>
                            </a>
                            <?php endif; ?>
                            <?php if ($userData['instagram_url']): ?>
                            <a href="<?php echo htmlspecialchars($userData['instagram_url']); ?>" class="social-link" target="_blank">
                                <i class="bi bi-instagram"></i>
                            </a>
                            <?php endif; ?>
                            <?php if ($userData['linkedin_url']): ?>
                            <a href="<?php echo htmlspecialchars($userData['linkedin_url']); ?>" class="social-link" target="_blank">
                                <i class="bi bi-linkedin"></i>
                            </a>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
                
                <?php 
                $galleryPhotos = array_filter($userData['photos'], function($photo) {
                    return $photo['photo_type'] === 'gallery';
                });
                if (!empty($galleryPhotos)): 
                ?>
                <div class="profile-card">
                    <h3 class="card-header"><i class="bi bi-images text-primary me-2"></i>Photos</h3>
                    <div class="photo-grid">
                        <?php foreach (array_slice($galleryPhotos, 0, 9) as $photo): ?>
                        <div class="photo-item">
                            <img src="/<?php echo htmlspecialchars($photo['photo_url']); ?>" alt="">
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <?php if (count($galleryPhotos) > 9): ?>
                    <a href="<?php echo APP_URL; ?>/profile/photos.php?id=<?php echo $viewingUserId; ?>" class="see-all-link">
                        See All Photos (<?php echo count($galleryPhotos); ?>)
                    </a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                
            </div>
        </div>
        
    </div>
</div>

<script src="<?php echo APP_URL; ?>/profile/js/profile.js"></script>

<?php 
include_once __DIR__ . '/../includes/footer.php';
include_once __DIR__ . '/../includes/bottom-nav.php';
?>