# AIMS - Agricultural Information and Market Linkage System

## Complete Project Structure - PHP & Bootstrap Edition

### Project Overview

**Student**: Musumba Jonathan (21/BCS/010/UMC)  
**Institution**: Metropolitan International University  
**Location**: Luuka District, Uganda  
**Tech Stack**: PHP, MySQL, Bootstrap 5, PWA

---

## 1. PROJECT DIRECTORY STRUCTURE

aims-platform/
├── index.php
├── config/
│   ├── database.php
│   ├── config.php
│   ├── session.php
│   └── constants.php
│
├── includes/
│   ├── header.php
│   ├── footer.php
│   ├── navbar.php
│   ├── sidebar.php
│   ├── bottom-nav.php
│   └── meta.php
│
├── auth/
│   ├── login.php
│   ├── register.php
│   ├── verify-otp.php
│   ├── logout.php
│   └── forgot-password.php
│
├── dashboard/
│   ├── index.php
│   ├── farmer-dashboard.php
│   ├── stats.php
│   └── widgets/
│       ├── weather-widget.php
│       ├── quick-actions.php
│       └── recent-activity.php
│
├── marketplace/
│   ├── index.php
│   ├── product-details.php
│   ├── add-product.php
│   ├── edit-product.php
│   ├── my-products.php
│   └── search.php
│
├── prices/
│   ├── index.php
│   ├── comparison.php
│   ├── alerts.php
│   └── set-alert.php
│
├── chat/
│   ├── index.php
│   ├── group-chat.php
│   ├── one-on-one.php
│   ├── create-group.php
│   └── ajax/
│       ├── send-message.php
│       ├── get-messages.php
│       └── load-chats.php
│
├── farm/
│   ├── index.php
│   ├── add-enterprise.php
│   ├── edit-enterprise.php
│   ├── transactions.php
│   ├── add-transaction.php
│   └── reports.php
│
├── ai-assistant/
│   ├── index.php
│   └── ajax/
│       └── chat.php
│
├── friends/
│   ├── index.php
│   ├── requests.php
│   ├── find-farmers.php
│   └── profile.php
│
├── learning/
│   ├── idex.phpn
│   ├── resources.php
│   ├── videos.php
│   └── best-practices.php
│
├── profile/
│   ├── index.php
│   ├── edit.php
│   ├── settings.php
│   └── change-password.php
│
├── admin/
│   ├── index.php
│   ├── users.php
│   ├── products.php
│   ├── prices.php
│   ├── reports.php
│   └── settings.php
│
├── api/
│   ├── auth.php
│   ├── products.php
│   ├── prices.php
│   ├── chat.php
│   ├── ai.php
│   └── notifications.php
│
├── ajax/
│   ├── notifications.php
│   ├── search.php
│   ├── upload.php
│   └── real-time-updates.php
│
├── classes/
│   ├── Database.php
│   ├── User.php
│   ├── Product.php
│   ├── Price.php
│   ├── Chat.php
│   ├── Message.php
│   ├── Farm.php
│   ├── Enterprise.php
│   ├── Transaction.php
│   ├── Friendship.php
│   ├── Notification.php
│   └── AIAssistant.php
│
├── assets/
│   ├── css/
│   │   ├── bootstrap.min.css
│   │   ├── bootstrap-icons.css
│   │   ├── custom.css
│   │   ├── responsive.css
│   │   ├── animations.css
│   │   └── themes/
│   │       ├── light.css
│   │       └── dark.css
│   ├── js/
│   │   ├── bootstrap.bundle.min.js
│   │   ├── jquery.min.js
│   │   ├── sweetalert2.min.js
│   │   ├── toastify.min.js
│   │   ├── Chart.min.js
│   │   ├── app.js
│   │   ├── notifications.js
│   │   ├── chat.js
│   │   ├── pwa.js
│   │   └── ajax-handler.js
│   ├── images/
│   │   ├── logo.png
│   │   ├── placeholder.jpg
│   │   └── icons/
│   ├── uploads/
│   │   ├── products/
│   │   ├── profiles/
│   │   └── chat/
│   └── fonts/
|──buyer/
|──extension/
│
├── pwa/
│   ├── manifest.json
│   ├── service-worker.js
│   ├── offline.html
│   └── icons/
│       ├── icon-72x72.png
│       ├── icon-96x96.png
│       ├── icon-128x128.png
│       ├── icon-144x144.png
│       ├── icon-152x152.png
│       ├── icon-192x192.png
│       ├── icon-384x384.png
│       └── icon-512x512.png
│
├── database/
│   ├── aims_database.sql
│   ├── migrations/
│   └── seeds/
│
├── vendor/
│   └── (Composer dependencies)
│
├── .htaccess
├── .env
├── composer.json
└── README.md

---

## 2. DATABASE SCHEMA (MySQL)

### Users Table

`sql
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    phone_number VARCHAR(15) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    full_name VARCHAR(100) NOT NULL,
    district VARCHAR(50) DEFAULT 'Luuka',
    subcounty VARCHAR(50),
    parish VARCHAR(50),
    village VARCHAR(50),
    language_preference ENUM('en', 'lusoga') DEFAULT 'en',
    profile_picture VARCHAR(255),
    user_type ENUM('farmer', 'buyer', 'extension_officer', 'admin') DEFAULT 'farmer',
    is_verified TINYINT(1) DEFAULT 0,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_login TIMESTAMP NULL,
    INDEX idx_phone (phone_number),
    INDEX idx_user_type (user_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Farms Table

`sql
CREATE TABLE farms (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    farm_name VARCHAR(100),
    total_acres DECIMAL(10,2),
    location_lat DECIMAL(10,8),
    location_lng DECIMAL(11,8),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Enterprises Table

`sql
CREATE TABLE enterprises (
    id INT AUTO_INCREMENT PRIMARY KEY,
    farm_id INT NOT NULL,
    enterprise_type ENUM('sugar_cane', 'poultry', 'maize', 'vegetables', 'other') NOT NULL,
    size_acres DECIMAL(10,2),
    planting_date DATE,
    expected_harvest_date DATE,
    status ENUM('planning', 'planted', 'growing', 'harvesting', 'completed') DEFAULT 'planning',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (farm_id) REFERENCES farms(id) ON DELETE CASCADE,
    INDEX idx_farm (farm_id),
    INDEX idx_type (enterprise_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Products Table

`sql
CREATE TABLE products (
    id INT AUTO_INCREMENT PRIMARY KEY,
    seller_id INT NOT NULL,
    enterprise_id INT,
    product_name VARCHAR(100) NOT NULL,
    category ENUM('sugar_cane', 'poultry', 'maize', 'vegetables', 'other') NOT NULL,
    description TEXT,
    quantity DECIMAL(10,2) NOT NULL,
    unit VARCHAR(20) NOT NULL,
    price_per_unit DECIMAL(10,2) NOT NULL,
    location VARCHAR(100),
    images JSON,
    status ENUM('available', 'sold', 'reserved') DEFAULT 'available',
    views INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (seller_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (enterprise_id) REFERENCES enterprises(id) ON DELETE SET NULL,
    INDEX idx_seller (seller_id),
    INDEX idx_category (category),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Market Prices Table

``sql
CREATE TABLE market_prices (
    id INT AUTO_INCREMENT PRIMARY KEY,
    product_type VARCHAR(50) NOT NULL,
    market_location VARCHAR(100) NOT NULL,
    price DECIMAL(10,2) NOT NULL,
    unit VARCHAR(20) NOT NULL,
    source VARCHAR(100),
    price_date DATE NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_product_date (product_type, price_date),
    INDEX idx_location (market_location)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Chats Table

``sql
CREATE TABLE chats (
    id INT AUTO_INCREMENT PRIMARY KEY,
    chat_type ENUM('group', 'one_to_one') NOT NULL,
    chat_name VARCHAR(100),
    chat_avatar VARCHAR(255),
    created_by INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_type (chat_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Chat Members Table

``sql
CREATE TABLE chat_members (
    id INT AUTO_INCREMENT PRIMARY KEY,
    chat_id INT NOT NULL,
    user_id INT NOT NULL,
    role ENUM('admin', 'member') DEFAULT 'member',
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_read_at TIMESTAMP NULL,
    FOREIGN KEY (chat_id) REFERENCES chats(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_membership (chat_id, user_id),
    INDEX idx_user (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Messages Table

``sql
CREATE TABLE messages (
    id INT AUTO_INCREMENT PRIMARY KEY,
    chat_id INT NOT NULL,
    sender_id INT NOT NULL,
    message_type ENUM('text', 'image', 'document', 'system') DEFAULT 'text',
    content TEXT NOT NULL,
    media_url VARCHAR(255),
    is_system_message TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (chat_id) REFERENCES chats(id) ON DELETE CASCADE,
    FOREIGN KEY (sender_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_chat_date (chat_id, created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Friendships Table

``sql
CREATE TABLE friendships (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    friend_id INT NOT NULL,
    status ENUM('pending', 'accepted', 'blocked') DEFAULT 'pending',
    requested_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    accepted_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (friend_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_friendship (user_id, friend_id),
    INDEX idx_user_status (user_id, status),
    CHECK (user_id != friend_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Transactions Table

``sql
CREATE TABLE transactions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    enterprise_id INT,
    transaction_type ENUM('income', 'expense') NOT NULL,
    category VARCHAR(50),
    amount DECIMAL(10,2) NOT NULL,
    description TEXT,
    transaction_date DATE NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (enterprise_id) REFERENCES enterprises(id) ON DELETE SET NULL,
    INDEX idx_user_date (user_id, transaction_date),
    INDEX idx_type (transaction_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Notifications Table

`sql
CREATE TABLE notifications (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    title VARCHAR(100) NOT NULL,
    message TEXT NOT NULL,
    type ENUM('info', 'success', 'warning', 'danger') DEFAULT 'info',
    icon VARCHAR(50),
    link VARCHAR(255),
    is_read TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_read (user_id, is_read),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### AI Conversations Table

`sql
CREATE TABLE ai_conversations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    message TEXT NOT NULL,
    response TEXT NOT NULL,
    language VARCHAR(10),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

### Price Alerts Table

``sql
CREATE TABLE price_alerts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    product_type VARCHAR(50) NOT NULL,
    target_price DECIMAL(10,2) NOT NULL,
    condition ENUM('above', 'below') NOT NULL,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_active (user_id, is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

---

## 3. TECHNOLOGY STACK

### Backend

- **Language**: PHP 8.1+
- **Database**: MySQL 8.0+
- **Session Management**: PHP Sessions
- **Authentication**: Password hashing with bcrypt
- **File Upload**: PHP native with validation
- **API**: RESTful endpoints

### Frontend

- **Framework**: Bootstrap 5.3
- **Icons**: Bootstrap Icons + Font Awesome 6
- **Notifications**: SweetAlert2 + Toastify
- **Charts**: Chart.js
- **JavaScript**: Vanilla JS + jQuery
- **AJAX**: jQuery AJAX

### Libraries & Plugins

- **Composer Packages**:
  - PHPMailer (email)
  - AfricasTalking SDK (SMS)
  - OpenAI PHP Client (AI)
  - Intervention Image (image processing)
  
### PWA

- **Service Worker**: Workbox
- **Offline Support**: IndexedDB
- **Push Notifications**: Web Push API

---

## 4. UI COMPONENTS & LIBRARIES

### CSS Libraries

``html
<!-- Bootstrap 5.3 -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">

<!-- Bootstrap Icons -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">

<!-- Font Awesome 6 -->
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css" rel="stylesheet">

<!-- Toastify (Beautiful Notifications) -->
<link href="https://cdn.jsdelivr.net/npm/toastify-js/src/toastify.min.css" rel="stylesheet">

<!-- Animate.css (Animations) -->
<link href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css" rel="stylesheet">

### JavaScript Libraries

`html
<!-- jQuery -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>

<!-- Bootstrap Bundle (includes Popper) -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>

<!-- SweetAlert2 (Beautiful Alerts) -->
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<!-- Toastify (Toast Notifications) -->
<script src="https://cdn.jsdelivr.net/npm/toastify-js"></script>

<!-- Chart.js (Charts) -->
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>

<!-- Pusher (Real-time) -->
<script src="https://js.pusher.com/8.2.0/pusher.min.js"></script>

---

## 5. NOTIFICATION SYSTEM

### Types of Notifications

1. **Toast Notifications** (Toastify)
2. **Alert Modals** (SweetAlert2)
3. **Badge Notifications** (Bootstrap)
4. **Push Notifications** (PWA)
5. **In-App Notifications** (Database-driven)

---

## 6. RESPONSIVE DESIGN

### Breakpoints (Bootstrap 5)

- **xs**: < 576px (Mobile)
- **sm**: ≥ 576px (Mobile landscape)
- **md**: ≥ 768px (Tablet)
- **lg**: ≥ 992px (Desktop)
- **xl**: ≥ 1200px (Large desktop)
- **xxl**: ≥ 1400px (Extra large)

### Mobile-First Features

- Bottom navigation for mobile
- Swipeable cards
- Touch-friendly buttons (min 44px)
- Collapsible menus
- Infinite scroll for lists

---

## 7. ENVIRONMENT SETUP (.env)

``env

# Database

DB_HOST=localhost
DB_NAME=aims_db
DB_USER=root
DB_PASS=

# App Settings

APP_NAME=AIMS
APP_URL=<http://localhost/aims>
APP_ENV=development
APP_DEBUG=true

# Security

SESSION_LIFETIME=7200
ENCRYPTION_KEY=your_encryption_key

# OpenAI

OPENAI_API_KEY=your_openai_api_key

# Africa's Talking

AT_USERNAME=your_username
AT_API_KEY=your_api_key

# File Upload

MAX_FILE_SIZE=5242880
ALLOWED_EXTENSIONS=jpg,jpeg,png,pdf,doc,docx

# Email

SMTP_HOST=smtp.gmail.com
SMTP_PORT=587
SMTP_USER=<noreplyaims@gmail.com>
SMTP_PASS=your_password
SMTP_FROM=<noreply@aims.com>

# PWA

PWA_NAME=AIMS
PWA_SHORT_NAME=AIMS
PWA_THEME_COLOR=#059669
PWA_BG_COLOR=#ffffff

---

## 8. COMPOSER.JSON

``json
{
    "name": "musumba/aims",
    "description": "Agricultural Information and Market Linkage System",
    "require": {
        "php": ">=8.1",
        "phpmailer/phpmailer": "^6.8",
        "africastalking/africastalking": "^3.0",
        "openai-php/client": "^0.7",
        "intervention/image": "^2.7",
        "vlucas/phpdotenv": "^5.5"
    },
    "autoload": {
        "psr-4": {
            "App\\": "classes/"
        }
    }
}

---

## 9. KEY PHP CLASSES STRUCTURE

### Database.php

`php
<?php
class Database {
    private $host = DB_HOST;
    private $db_name = DB_NAME;
    private $username = DB_USER;
    private $password = DB_PASS;
    public $conn;

    public function getConnection() {
        $this->conn = null;
        try {
            $this->conn = new PDO(
                "mysql:host=" . $this->host . ";dbname=" . $this->db_name,
                $this->username,
                $this->password
            );
            $this->conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->conn->exec("set names utf8mb4");
        } catch(PDOException $e) {
            echo "Connection error: " . $e->getMessage();
        }
        return $this->conn;
    }
}
