-- =====================================================
-- COMPLETE VILLAGE MERGE SCRIPT
-- Merge all villages from LUPA database into AIMS database
-- =====================================================
-- This script adds all missing villages from the LUPA database
-- to the AIMS database, properly mapped to their parishes
-- =====================================================

USE aims_db;

START TRANSACTION;

-- =====================================================
-- STEP 1: CREATE PROCEDURE FOR SYSTEMATIC VILLAGE INSERTION
-- =====================================================

DELIMITER $$

DROP PROCEDURE IF EXISTS InsertVillageIfNotExists$$

CREATE PROCEDURE InsertVillageIfNotExists(
    IN p_district_name VARCHAR(100),
    IN p_subcounty_name VARCHAR(100),
    IN p_parish_name VARCHAR(100),
    IN p_village_name VARCHAR(100),
    IN p_village_code VARCHAR(10)
)
BEGIN
    DECLARE v_parish_id INT;
    
    -- Get parish ID based on the hierarchy
    SELECT p.id INTO v_parish_id
    FROM parishes p
    JOIN subcounties s ON p.subcounty_id = s.id
    JOIN districts d ON s.district_id = d.id
    WHERE d.district_name = p_district_name
        AND s.subcounty_name = p_subcounty_name
        AND p.parish_name = p_parish_name
    LIMIT 1;
    
    -- Insert village if parish exists and village doesn't exist
    IF v_parish_id IS NOT NULL THEN
        INSERT IGNORE INTO villages (parish_id, village_name, village_code)
        VALUES (v_parish_id, p_village_name, p_village_code);
    END IF;
END$$

DELIMITER ;

-- =====================================================
-- STEP 2: INSERT ALL VILLAGES FROM LUPA DATABASE
-- =====================================================

-- LUUKA DISTRICT VILLAGES
-- Bulongo Subcounty
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Budhabangula', 'Budhabangula A', 'BDB-A');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Budhabangula', 'Budhabangula B', 'BDB-B');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Budhabangula', 'Kibuutu', 'KBT');

CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bugonyoka', 'Bugonyoka A', 'BGY-A');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bugonyoka', 'Bugonyoka B', 'BGY-B');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bugonyoka', 'Buzaaya', 'BZY');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bugonyoka', 'Nakisenyi A', 'NKS-A');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bugonyoka', 'Nakisenyi B', 'NKS-B');

CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bukendi', 'Bugabula A', 'BGB-A');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bukendi', 'Bugabula B', 'BGB-B');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bukendi', 'Bukendi', 'BKD');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bukendi', 'Nabitama A', 'NBT-A');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bukendi', 'Nabitama B', 'NBT-B');

CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bulongo', 'Buwaiswa', 'BWS');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bulongo', 'Kamwirungu A', 'KMW-A');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Bulongo', 'Kamwirungu B', 'KMW-B');

CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Nakabugu', 'Buseete', 'BST');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Nakabugu', 'Buyunze A', 'BYZ-A');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Nakabugu', 'Buyunze B', 'BYZ-B');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Nakabugu', 'Nakabugu A', 'NKB-A');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Nakabugu', 'Nakabugu B', 'NKB-B');

CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Namalemba', 'Bulike', 'BLK');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Namalemba', 'Busala', 'BSL');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Namalemba', 'Kasozi', 'KSZ');
CALL InsertVillageIfNotExists('Luuka', 'Bulongo', 'Namalemba', 'Namalemba', 'NML');

-- Ikumbya Subcounty
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Bunafu', 'Bunafu A', 'BNF-A');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Bunafu', 'Bunafu B', 'BNF-B');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Bunafu', 'Nabisira', 'NBS');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Bunafu', 'Nawanyago', 'NWG');

CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Ikumbya', 'Buwutu', 'BWT');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Ikumbya', 'Ikumbya', 'IKM');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Ikumbya', 'Nsambya', 'NSM');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Ikumbya', 'Wandago', 'WND');

CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Inuula', 'Budhuuba A', 'BDH-A');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Inuula', 'Budhuuba B', 'BDH-B');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Inuula', 'Bugambo', 'BGM');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Inuula', 'Inula', 'INL');

CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Nawaka', 'Bugonza', 'BGZ');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Nawaka', 'Kawanga', 'KWG');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Nawaka', 'Nantamali', 'NTM');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Nawaka', 'Nawaka A', 'NWK-A');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Nawaka', 'Nawaka B', 'NWK-B');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Nawaka', 'Nawaka C (Buyego)', 'NWK-C');

CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Ntayigirwa', 'Bukobbo', 'BKB');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Ntayigirwa', 'Bulike', 'BLK');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Ntayigirwa', 'Idoome East', 'IDM-E');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Ntayigirwa', 'Idoome West', 'IDM-W');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Ntayigirwa', 'Nabitende', 'NBT');
CALL InsertVillageIfNotExists('Luuka', 'Ikumbya', 'Ntayigirwa', 'Ntayigirwa', 'NTY');

-- Luuka Town Council
CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Busimawu Ward', 'Budhumbwire', 'BDW');
CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Busimawu Ward', 'Busimawu Zone', 'BSM');
CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Busimawu Ward', 'Waigugwe Zone', 'WGW');

CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Busonga Ward', 'Bukabooli Zone', 'BKB');
CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Busonga Ward', 'Busonga Zone', 'BSG');

CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Kitwekyambogo Ward', 'Mbogo Zone', 'MBG');
CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Kitwekyambogo Ward', 'Nawamwena', 'NWM');
CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Kitwekyambogo Ward', 'Nawansudde Zone', 'NWS');

CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Kiyunga Ward', 'Kiyunga Central Zone', 'KYC');
CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Kiyunga Ward', 'Kiyunga Hospital Zone', 'KYH');
CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Kiyunga Ward', 'Kiyunga Industrial zone', 'KYI');

CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Lwada Ward', 'Buwambuzi', 'BWZ');
CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Lwada Ward', 'Lwanda', 'LWD');
CALL InsertVillageIfNotExists('Luuka', 'Luuka Town Council', 'Lwada Ward', 'Ntandagwe', 'NTD');

-- Bukanga Subcounty
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Budondo', 'Budondo A', 'BDD-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Budondo', 'Budondo B', 'BDD-B');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Budondo', 'Bugoba', 'BGB');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Budondo', 'Kimanto A', 'KMT-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Budondo', 'Kimanto B', 'KMT-B');

CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Buwologoma', 'Bugogo', 'BGG');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Buwologoma', 'Bukaade', 'BKD');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Buwologoma', 'Bukwanga', 'BKW');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Buwologoma', 'Buwologoma Central', 'BWL');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Buwologoma', 'Katalakabi', 'KTL');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Buwologoma', 'Nabinoni', 'NBN');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Buwologoma', 'Nakamini', 'NKM');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Buwologoma', 'Ndhoya A', 'NDH-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Buwologoma', 'Ndhoya B', 'NDH-B');

CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Kiroba', 'Bighunu', 'BGH');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Kiroba', 'Bulonde', 'BLD');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Kiroba', 'Busandha', 'BSD');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Kiroba', 'Kiroba-Budoma', 'KRB');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Kiroba', 'Kisimba', 'KSM');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Kiroba', 'Nawantale', 'NWT');

CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Nabubya', 'Budoma A', 'BDM-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Nabubya', 'Budoma B', 'BDM-B');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Nabubya', 'Budoma C', 'BDM-C');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Nabubya', 'Bunhirira', 'BNH');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Nabubya', 'Nabubya A', 'NBB-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Nabubya', 'Nabubya B', 'NBB-B');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Nabubya', 'Nakabondo', 'NKB');

CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Namukubembe', 'Bukanga-Bukendi', 'BKB');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Namukubembe', 'Bulwasira', 'BWS');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Namukubembe', 'Bumanha A', 'BMH-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Namukubembe', 'Bumanha B', 'BMH-B');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Namukubembe', 'Bumanha Central', 'BMH-C');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Namukubembe', 'Kantenga', 'KTG');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Namukubembe', 'Namukubembe A', 'NMK-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukanga', 'Namukubembe', 'Namukubembe B', 'NMK-B');

-- Bulanga Town Council
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Bulanga Ward', 'Bulanga Estate', 'BLE');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Bulanga Ward', 'Kasokoso', 'KSK');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Bulanga Ward', 'Kataike', 'KTK');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Bulanga Ward', 'Kiwumbi', 'KWM');

CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Itwe Ward', 'Bugwanala A', 'BGW-A');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Itwe Ward', 'Bugwanala B', 'BGW-B');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Itwe Ward', 'Itwe', 'ITW');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Itwe Ward', 'Tagoole', 'TGL');

CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Mawundo Ward', 'Budhebera', 'BDB');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Mawundo Ward', 'Kyankuzi', 'KYK');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Mawundo Ward', 'Mufuwa', 'MFW');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Mawundo Ward', 'Nabukalu', 'NBK');

CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Nantamu Ward', 'Abalawuwe', 'ABL');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Nantamu Ward', 'Lumbuye', 'LMB');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Nantamu Ward', 'Luswiga', 'LSW');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Nantamu Ward', 'Nantamu', 'NTM');

CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Walibo Ward', 'Kisansa', 'KSS');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Walibo Ward', 'Luwangula', 'LWG');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Walibo Ward', 'Walibo', 'WLB');
CALL InsertVillageIfNotExists('Luuka', 'Bulanga Town Council', 'Walibo Ward', 'Walibo Lulenzi', 'WLL');

-- Busalamu Town Council
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu East', 'Botondola B', 'BTB');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu East', 'Busalamu Central', 'BSC');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu East', 'Butondolo A', 'BTA');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu East', 'Kidyope', 'KDY');

CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu North', 'Busiringe A', 'BSR-A');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu North', 'Busiringe B', 'BSR-B');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu North', 'Kiringo A', 'KRG-A');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu North', 'Kiringo B', 'KRG-B');

CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu South', 'Kigulamo A', 'KGL-A');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu South', 'Kigulamo B', 'KGL-B');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu South', 'Lukunhu A', 'LKH-A');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu South', 'Lukunhu B', 'LKH-B');

CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu West', 'Lukotaime A', 'LKT-A');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu West', 'Lukotaime B', 'LKT-B');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu West', 'Lukotaime C', 'LKT-C');
CALL InsertVillageIfNotExists('Luuka', 'Busalamu Town Council', 'Busalamu West', 'Lukotaime D', 'LKT-D');

-- Irongo Subcounty
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Irongo', 'Bufumba', 'BFM');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Irongo', 'Buwala', 'BWL');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Irongo', 'Gansembye', 'GSB');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Irongo', 'Iganga A', 'IGA-A');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Irongo', 'Iganga B', 'IGA-B');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Irongo', 'Irongo', 'IRG');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Irongo', 'Naimuli A', 'NML-A');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Irongo', 'Naimuli B', 'NML-B');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Irongo', 'Nsirira', 'NSR');

CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kalyowa', 'Bukyamata', 'BKY');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kalyowa', 'Kalyowa A', 'KLY-A');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kalyowa', 'Kalyowa B', 'KLY-B');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kalyowa', 'Kigunga', 'KGG');

CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kibinga', 'Kantenga', 'KTG');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kibinga', 'Kibinga A', 'KBG-A');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kibinga', 'Kibinga B', 'KBG-B');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kibinga', 'Nakavuma', 'NKV');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kibinga', 'Nkandakulyowa', 'NKK');

CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kyanvuma', 'Buwaigala', 'BWG');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kyanvuma', 'Kiwalazi', 'KWL');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kyanvuma', 'Kizigo', 'KZG');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kyanvuma', 'Nakabaale B', 'NKB-B');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Kyanvuma', 'Nakasedhere', 'NKS');

CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Nawanyago', 'Bubiro', 'BBR');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Nawanyago', 'Bugaga A', 'BGA-A');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Nawanyago', 'Bugaga B', 'BGA-B');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Nawanyago', 'Bugobi', 'BGB');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Nawanyago', 'Butogonya', 'BTG');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Nawanyago', 'Buyemba', 'BYM');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Nawanyago', 'Luzinga Park', 'LZP');
CALL InsertVillageIfNotExists('Luuka', 'Irongo', 'Nawanyago', 'Nawampiti', 'NWP');

-- Kyanvuma Town Council
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Buniko Ward', 'Buniko A', 'BNK-A');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Buniko Ward', 'Buniko B', 'BNK-B');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Buniko Ward', 'Mawale', 'MWL');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Buniko Ward', 'Mboga', 'MBG');

CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Magada Ward', 'Kasokoso', 'KSK');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Magada Ward', 'Kibuye LC1', 'KBY');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Magada Ward', 'Kyanvuma Trading Center', 'KYT');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Magada Ward', 'Old Market', 'OLD');

CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nakabaale Ward', 'Bondo', 'BND');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nakabaale Ward', 'Nakabaale A', 'NKB-A');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nakabaale Ward', 'Nakabaale scheme', 'NKB-S');

CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nakabambwe Ward', 'Kigaito', 'KGT');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nakabambwe Ward', 'Kyanvuma', 'KYV');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nakabambwe Ward', 'Nabikamba', 'NBK');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nakabambwe Ward', 'Nakabambwe', 'NKB');

CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nsimakatono Ward', 'Budugu', 'BDG');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nsimakatono Ward', 'Bulinda', 'BLD');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nsimakatono Ward', 'Mpala', 'MPL');
CALL InsertVillageIfNotExists('Luuka', 'Kyanvuma Town Council', 'Nsimakatono Ward', 'Nsiima Trading Centre', 'NSM');

-- Nawampiti Subcounty
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Bugomba', 'Bugomba', 'BGM');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Bugomba', 'Buwanda', 'BWD');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Bugomba', 'Nawandyo', 'NWD');

CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Buyoola', 'Buyoola', 'BYL');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Buyoola', 'Ikonia A', 'IKA-A');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Buyoola', 'Ikonia Busige', 'IKB');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Buyoola', 'Ikonia Central', 'IKC');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Buyoola', 'Nakyere', 'NKY');

CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nakiswiga', 'Buwamwa', 'BWM');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nakiswiga', 'Nabikuyi', 'NBK');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nakiswiga', 'Nakiswiga A', 'NKS-A');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nakiswiga', 'Nakiswiga B', 'NKS-B');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nakiswiga', 'Namagera', 'NMG');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nakiswiga', 'Njababona', 'NJB');

CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nawampiti', 'Buzimba', 'BZM');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nawampiti', 'Kituuto A', 'KTT-A');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nawampiti', 'Kituuto B', 'KTT-B');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nawampiti', 'Nawampiti-Bukyangwa', 'NWB');

CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nawankompe', 'Kasozi', 'KSZ');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nawankompe', 'Nawankompe A', 'NWK-A');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nawankompe', 'Nawankompe B', 'NWK-B');
CALL InsertVillageIfNotExists('Luuka', 'Nawampiti', 'Nawankompe', 'Walulagaba', 'WLG');

-- Waibuga Subcounty
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Busiiro', 'Bulindi', 'BLD');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Busiiro', 'Busiiro', 'BSR');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Busiiro', 'Busiiro TC', 'BST');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Busiiro', 'Bwamba', 'BWM');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Busiiro', 'Kyamawundo', 'KYM');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Busiiro', 'Nanvunano A', 'NVN-A');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Busiiro', 'Nanvunano B', 'NVN-B');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Busiiro', 'Nanvunano C', 'NVN-C');

CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Butimbwa', 'Bukyogo', 'BKY');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Butimbwa', 'Butimbwa A', 'BTM-A');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Butimbwa', 'Butimbwa B', 'BTM-B');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Butimbwa', 'Ikonko', 'IKK');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Butimbwa', 'Lwanika', 'LWN');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Butimbwa', 'Namakakale A', 'NMK-A');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Butimbwa', 'Namakakale B', 'NMK-B');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Butimbwa', 'Waibuga A', 'WBG-A');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Butimbwa', 'Waibuga B', 'WBG-B');

CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Itakaibolu', 'Bukapala', 'BKP');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Itakaibolu', 'Buwiiri', 'BWR');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Itakaibolu', 'Itakaibolu A', 'ITK-A');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Itakaibolu', 'Itakaibolu B', 'ITK-B');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Itakaibolu', 'Kigaya', 'KGY');

CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Lwaki', 'Kiwanyi', 'KWY');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Lwaki', 'Lwaki', 'LWK');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Lwaki', 'Maumo', 'MAM');
CALL InsertVillageIfNotExists('Luuka', 'Waibuga', 'Lwaki', 'Namadope', 'NMD');

-- Bukoma Subcounty
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Bukoma', 'Bukooma B', 'BKM-B');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Bukoma', 'Bukooma a', 'BKM-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Bukoma', 'Misita', 'MST');

CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Bukyangwa', 'Budhaana A', 'BDH-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Bukyangwa', 'Budhaana B', 'BDH-B');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Bukyangwa', 'Bukyangwa A', 'BKY-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Bukyangwa', 'Bukyangwa B', 'BKY-B');

CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Naigobya', 'Naigobya A', 'NGB-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Naigobya', 'Naigobya B', 'NGB-B');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Naigobya', 'Naigobya T/C', 'NGB-T');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Naigobya', 'Nairika East', 'NRK-E');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Naigobya', 'Nairika West', 'NRK-W');

CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namansenda', 'Bulalu', 'BLL');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namansenda', 'Busaku', 'BSK');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namansenda', 'Kirimwa A', 'KRM-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namansenda', 'Kirimwa B', 'KRM-B');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namansenda', 'Nabimogo A', 'NBM-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namansenda', 'Nabimogo B', 'NBM-B');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namansenda', 'Namansenda A', 'NMS-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namansenda', 'Namansenda B', 'NMS-B');

CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namulanda', 'Gwembuzi A', 'GWM-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namulanda', 'Gwembuzi B', 'GWM-B');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namulanda', 'Namulanda A', 'NML-A');
CALL InsertVillageIfNotExists('Luuka', 'Bukoma', 'Namulanda', 'Namulanda B', 'NML-B');

-- Bukoova Town Council
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukhana Ward', 'Bumyuka', 'BMY');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukhana Ward', 'Buyego', 'BYG');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukhana Ward', 'Ibanda Yonna', 'IBY');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukhana Ward', 'Isaka Buga', 'ISB');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukhana Ward', 'Luganda', 'LGD');

CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukoova Central', 'Agape Zone', 'AGP');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukoova Central', 'Central Zone', 'CTZ');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukoova Central', 'Kisambira Zone', 'KSM');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukoova Central', 'Kunya Zone', 'KNY');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukoova Central', 'Yonna Zone', 'YNN');

CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukoova Rural', 'Bufumba Zone', 'BFM');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukoova Rural', 'Bugumba Zone', 'BGM');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukoova Rural', 'Kilangila Zone', 'KLG');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bukoova Rural', 'Kiteso Zone', 'KTS');

CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bunabala Ward', 'Bugaya Zone', 'BGY');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bunabala Ward', 'Bukoova PS Zone', 'BKP');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bunabala Ward', 'Bususwa Zone', 'BSS');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bunabala Ward', 'Busuuyi Zone', 'BSY');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Bunabala Ward', 'Buyingo Zone', 'BYG');

CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Busanda Ward', 'Budoola Zone', 'BDL');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Busanda Ward', 'Bwinike Zone', 'BWN');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Busanda Ward', 'Central Zone', 'CTZ');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Busanda Ward', 'Landi Zone', 'LND');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Busanda Ward', 'Ngalambi Zone', 'NGL');

CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Butaserwa Ward', 'Bufumba Zone', 'BFM');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Butaserwa Ward', 'Bugaga Zone', 'BGA');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Butaserwa Ward', 'Bulangira Zone', 'BLG');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Butaserwa Ward', 'Central Zone', 'CTZ');

CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Buyoga Ward', 'Bufumba Zone', 'BFM');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Buyoga Ward', 'Bukyega Zone', 'BKY');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Buyoga Ward', 'Busuubo Zone', 'BSB');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Buyoga Ward', 'Buyingo Zone', 'BYG');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Buyoga Ward', 'Teso Zone', 'TSO');

CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Makkutu Ward', 'Bubiro Zone', 'BBR');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Makkutu Ward', 'Budhugu Zone', 'BDG');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Makkutu Ward', 'Buyingo Zone', 'BYG');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Makkutu Ward', 'Bwanda Zone', 'BWD');

CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nabyoto Ward', 'Bukama Zone', 'BKM');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nabyoto Ward', 'Central Zone', 'CTZ');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nabyoto Ward', 'Kisita Zone', 'KST');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nabyoto Ward', 'Mutyagoma Zone', 'MTG');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nabyoto Ward', 'Railway Zone', 'RWY');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nabyoto Ward', 'Wakasa Zone', 'WKS');

CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nawansega Ward', 'Bukubembe/Buguru Zone', 'BKB');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nawansega Ward', 'Bukwanga Zone', 'BKW');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nawansega Ward', 'Bulondo Zone', 'BLD');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nawansega Ward', 'Buyawuka Zone', 'BYW');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nawansega Ward', 'Central Zone', 'CTZ');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nawansega Ward', 'Kiseege Zone', 'KSG');
CALL InsertVillageIfNotExists('Luuka', 'Bukoova Town Council', 'Nawansega Ward', 'Landi Zone', 'LND');

-- =====================================================
-- STEP 4: INSERT IGANGA DISTRICT VILLAGES
-- =====================================================
-- (The LUPA database has extensive Iganga village data)
-- I'll create a comprehensive section for all Iganga villages

-- NABITENDE Subcounty Villages
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'BUGONO', 'BUGONO', 'BGN-1');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'BUGONO', 'NABINONYI', 'NBN');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'BUGONO', 'BUGONO B', 'BGN-B');

CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'ITANDA', 'BUVULE', 'BVL');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'ITANDA', 'ITANDA - BUWEIRA', 'ITB');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'ITANDA', 'KINU', 'KNU');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'ITANDA', 'BUKAIGO', 'BKG');

CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KABIRA-ITUBA', 'BULIGANHA', 'BLG');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KABIRA-ITUBA', 'ITUBA A', 'ITB-A');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KABIRA-ITUBA', 'KABIRA A', 'KBR-A');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KABIRA-ITUBA', 'ITUBA BUNIGULE', 'ITB-B');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KABIRA-ITUBA', 'KABIRA B', 'KBR-B');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KABIRA-ITUBA', 'BUIGANHA B', 'BIG-B');

CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KASAMBIKA', 'BUWEREMPE', 'BWR');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KASAMBIKA', 'KASAMBIKA I', 'KSM-1');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KASAMBIKA', 'KASAMBIKA II', 'KSM-2');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KASAMBIKA', 'NAMUSIISI I', 'NMS-1');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'KASAMBIKA', 'NAMUSIISI II', 'NMS-2');

CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'NABITENDE', 'BUBEKO', 'BBK');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'NABITENDE', 'KALUNGAMI \'A\'', 'KLG-A');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'NABITENDE', 'KALUNGAMI \'B\'', 'KLG-B');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'NABITENDE', 'NABITENDE', 'NBT-1');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'NABITENDE', 'BUTABALA', 'BTB');

CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'NALUKO', 'NALUKO', 'NLK-1');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'NALUKO', 'NAWANKWALE', 'NWK');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'NALUKO', 'NALUKO BUZIBIRILA', 'NLK-B');
CALL InsertVillageIfNotExists('Iganga', 'NABITENDE', 'NALUKO', 'NAWANKWALE BUGWERE', 'NWK-B');

-- NAMBALE Subcounty Villages
CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'MWIRA', 'KAMIRABIKOMO', 'KMR');
CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'MWIRA', 'MWIRA', 'MWR-1');
CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'MWIRA', 'NABITOVU', 'NBT-1');
CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'MWIRA', 'NABITOVU NORTH', 'NBT-N');

CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'NAMBALE', 'NAMBALE I A', 'NMB-1A');
CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'NAMBALE', 'NAMBALE I B', 'NMB-1B');
CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'NAMBALE', 'NAMBALE II', 'NMB-2');

CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'NASUTI', 'BUSIMBA', 'BSM');
CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'NASUTI', 'NABUKONE', 'NBK');
CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'NASUTI', 'NASUUTI NORTH', 'NST-N');
CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'NASUTI', 'NASUUTI SOUTH', 'NST-S');
CALL InsertVillageIfNotExists('Iganga', 'NAMBALE', 'NASUTI', 'KAKIRA', 'KKR');

-- NAMUNGALWE TOWN COUNCIL Villages
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNKESU WARD', 'BUBOGO A', 'BBG-A');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNKESU WARD', 'NABIKOOTE', 'NBK');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNKESU WARD', 'NAMUFUMA', 'NMF');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNKESU WARD', 'NAMUNKESU', 'NMK');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNKESU WARD', 'BUBOGO B', 'BBG-B');

CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNSAALA WARD', 'BUFUTULA A', 'BFT-A');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNSAALA WARD', 'NAMUNSAALA', 'NMS');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNSAALA WARD', 'BUFUTULA B', 'BFT-B');

CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAWANSEGA WARD', 'NAWANSEGA A', 'NWS-A');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAWANSEGA WARD', 'NAWANSEGA B', 'NWS-B');

CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'BULUMWAKI WARD', 'BULUMWAKI I', 'BLM-1');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'BULUMWAKI WARD', 'BULUMWAKI II', 'BLM-2');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'BULUMWAKI WARD', 'NSINZE', 'NSZ');

CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'MWENDANFUKO WARD', 'BULANGA', 'BLG');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'MWENDANFUKO WARD', 'MWENDANFUKO', 'MWF');

CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNGALWE WARD', 'KAWETE', 'KWT');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNGALWE WARD', 'NAMUNGALWE RURAL', 'NMG-R');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNGALWE WARD', 'NAMUNGALWE A', 'NMG-A');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNGALWE WARD', 'NAMUNGALWE B', 'NMG-B');

CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNKANAGA WARD', 'NAMUNKANAGA I', 'NMK-1');
CALL InsertVillageIfNotExists('Iganga', 'NAMUNGALWE TOWN COUNCIL', 'NAMUNKANAGA WARD', 'NAMUNKANAGA II', 'NMK-2');

-- NAWANDALA Subcounty Villages
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAWANGAIZA', 'BUGOLE A', 'BGL-A');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAWANGAIZA', 'BUGOLE B', 'BGL-B');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAWANGAIZA', 'BUGONGO', 'BGG-1');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAWANGAIZA', 'NAMABWERE I', 'NMB-1');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAWANGAIZA', 'NAMABWERE II', 'NMB-2');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAWANGAIZA', 'NAMUKUMYA', 'NMK');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAWANGAIZA', 'NAWANDALA', 'NWD');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAWANGAIZA', 'NAWANGAIZA II', 'NWG-2');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAWANGAIZA', 'BUSAKAIRE', 'BSK');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAWANGAIZA', 'BUGONGO B', 'BGG-B');

CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KIWANYI', 'KIWANYI I', 'KWY-1');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KIWANYI', 'KIWANYI II', 'KWY-2');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KIWANYI', 'BUSAALA', 'BSL');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KIWANYI', 'BUWUMBA', 'BWM');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KIWANYI', 'MADHIMASU', 'MDH');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KIWANYI', 'BUKYEEGA', 'BKY');

CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KYENDABAWALA', 'BUGAMBO', 'BGM');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KYENDABAWALA', 'BUNYOKANO', 'BNY');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KYENDABAWALA', 'BUSOIGI', 'BSG');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KYENDABAWALA', 'BUZAAYA', 'BZY');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KYENDABAWALA', 'IKUMBYA', 'IKM');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KYENDABAWALA', 'KABULI I', 'KBL-1');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KYENDABAWALA', 'KABULI II', 'KBL-2');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'KYENDABAWALA', 'KYENDABAWALA', 'KYB');

CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAMUSIISI', 'BUYUNGA', 'BYG');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAMUSIISI', 'MALOBI', 'MLB');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAMUSIISI', 'NAMUSIISI', 'NMS-1');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'NAMUSIISI', 'BUGANZA', 'BGZ');

CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'BUGONGO', 'KIRINGA A', 'KRG-A');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'BUGONGO', 'KIRINGA B', 'KRG-B');
CALL InsertVillageIfNotExists('Iganga', 'NAWANDALA', 'BUGONGO', 'NAWANGAIZA I', 'NWG-1');

-- KIDAAGO Subcounty Villages
CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'NAIBIRI', 'NAIBIRI CENTRAL', 'NBR-C');
CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'NAIBIRI', 'NAIBIRI NORTH', 'NBR-N');
CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'NAIBIRI', 'NAIBIRI SOUTH', 'NBR-S');

CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'KIDAAGO', 'KIDAAGO A', 'KDG-A');
CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'KIDAAGO', 'KIDAAGO B', 'KDG-B');
CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'KAZIGO', 'KAZIGO A', 'KZG-A');
CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'KAZIGO', 'KAZIGO B', 'KZG-B');

CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'NABITENDE', 'NABITENDE A', 'NBT-A');
CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'NABITENDE', 'NABITENDE B', 'NBT-B');
CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'NABITENDE', 'BUSABI ZONE', 'BSB');
CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'NABITENDE', 'BUKOSE ZONE', 'BKS');
CALL InsertVillageIfNotExists('Iganga', 'KIDAAGO', 'NABITENDE', 'BUYALE ZONE', 'BYL');

-- BULAMAGI Subcounty Villages
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BUKOYO', 'BUDHWEGE', 'BDW');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BUKOYO', 'BUKOYO', 'BKY-1');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BUKOYO', 'NAWANSINGE', 'NWS');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BUKOYO', 'BUKABISI', 'BKB');

CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BULAMAGI', 'BUBAKA', 'BBK');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BULAMAGI', 'BULAMAGI', 'BLG-1');

CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BULOWOOZA', 'KASOLO', 'KSL');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BULOWOOZA', 'BULOWOOZA', 'BLW-1');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BULOWOOZA', 'MADHIGANDERE', 'MDG');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BULOWOOZA', 'NGANGALI', 'NGL');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BULOWOOZA', 'WALUKUBA', 'WLK');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BULOWOOZA', 'MAWOITO', 'MWT-1');

CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BWANALIRA', 'BUDHEBERA', 'BDB');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BWANALIRA', 'IGULAMUBIRI', 'IGL');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BWANALIRA', 'BUWASA', 'BWS');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BWANALIRA', 'BUYUBU', 'BYB');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BWANALIRA', 'BWANALIRA', 'BWN-1');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'BWANALIRA', 'KINAWANSWA', 'KNW');

CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'IWAWU', 'BUWOYA', 'BWY');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'IWAWU', 'IWAWU', 'IWW-1');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'IWAWU', 'KYEMEIRE', 'KYM');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'IWAWU', 'MASABA', 'MSB');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'IWAWU', 'NAMASOGA', 'NMS');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'IWAWU', 'NAWANKOFU', 'NWK');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'IWAWU', 'NKAZIWERU', 'NKZ');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'IWAWU', 'NTINDA', 'NTD');
CALL InsertVillageIfNotExists('Iganga', 'BULAMAGI', 'IWAWU', 'NAMALEMBA', 'NML');

-- NAKALAMA Subcounty Villages
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKAYE', 'BUKAYE', 'BKY-1');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKAYE', 'BUWONGO', 'BWG');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKAYE', 'NAMPIRIKA A', 'NMP-A');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKAYE', 'NAMPIRIKA B', 'NMP-B');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKAYE', 'IGULUSA', 'IGL');

CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKOONA', 'BUPALA', 'BPL');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKOONA', 'KAKONGOKA', 'KKG-1');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKOONA', 'NAMUNDUDI A', 'NMD-A');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKOONA', 'NAMUNDUDI B', 'NMD-B');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKOONA', 'KAKONGOKA B', 'KKG-B');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKOONA', 'NAMUNDUDI C', 'NMD-C');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUKOONA', 'BUKALAMBWA', 'BKL');

CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUSEYI', 'BUSEYI A', 'BSY-A');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUSEYI', 'BUSEYI B', 'BSY-B');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'BUSEYI', 'BUSEYI CENTRAL', 'BSY-C');

CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'NAKALAMA', 'BUKOBOOLI', 'BKB');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'NAKALAMA', 'BUKYAYE', 'BKY');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'NAKALAMA', 'BUTAAMA', 'BTM');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'NAKALAMA', 'NABIRYE', 'NBR');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'NAKALAMA', 'NAKALAMA CENTRAL', 'NKL-C');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'NAKALAMA', 'NAKALAMA SOUTH', 'NKL-S');
CALL InsertVillageIfNotExists('Iganga', 'NAKALAMA', 'NAKALAMA', 'NAKALAMA NORTH', 'NKL-N');

-- NAKIGO Subcounty Villages
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BULUBANDI', 'BUGABWE', 'BGB');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BULUBANDI', 'BULUBANDI CENTRAL A', 'BLB-A');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BULUBANDI', 'NANDEKULA A', 'NND-A');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BULUBANDI', 'BULUBANDI CENTRAL B', 'BLB-B');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BULUBANDI', 'NANDEKULA B', 'NND-B');

CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BUNYAMA', 'BUKWAYA', 'BKW');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BUNYAMA', 'BUNYAMA', 'BNY-1');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BUNYAMA', 'KIBOYO', 'KBY');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BUNYAMA', 'NAMIRALI', 'NMR');

CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BUSOWOOBI', 'BUSOWOOBI- NYENGA', 'BSW-N');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BUSOWOOBI', 'BUSOWOOBI CENTRAL', 'BSW-C');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BUSOWOOBI', 'NABITOVU', 'NBT');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BUSOWOOBI', 'NAKIGO I', 'NKG-1');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BUSOWOOBI', 'NAKIGO II', 'NKG-2');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'BUSOWOOBI', 'KIWERERE', 'KWR');

CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'KABIRA', 'BULUZA', 'BLZ');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'KABIRA', 'KABIRA', 'KBR-1');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'KABIRA', 'NAWANZU', 'NWZ');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'KABIRA', 'NAMUSENWA', 'NMS');

CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'WAIRAMA', 'IZIMBA', 'IZM');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'WAIRAMA', 'KAKOMBO', 'KKM');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'WAIRAMA', 'NAKISENYI', 'NKS');
CALL InsertVillageIfNotExists('Iganga', 'NAKIGO', 'WAIRAMA', 'WAIRAMA', 'WRM-1');

-- NAWANYINGI Subcounty Villages
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'BUNYIRO', 'BUNYIIRO', 'BNY-1');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'BUNYIRO', 'BUWOLOMERA A', 'BWL-A');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'BUNYIRO', 'BUWOLOMER B', 'BWL-B');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'BUNYIRO', 'NAGADUDULA', 'NGD');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'BUNYIRO', 'BUKOSE', 'BKS');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'BUNYIRO', 'LWABAGA', 'LWB');

CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'MAGOGO', 'BUKONKO', 'BKK');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'MAGOGO', 'LWERERA', 'LWR');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'MAGOGO', 'MAGOGO', 'MGG-1');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'MAGOGO', 'NAKIRULWE', 'NKR');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'MAGOGO', 'NAWANDYO', 'NWD');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'MAGOGO', 'BUWANDE', 'BWD');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'MAGOGO', 'BUWENDA', 'BWN');

CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'NAWANYINGI', 'MAWAGALA', 'MWG');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'NAWANYINGI', 'NAWANKONGE', 'NWK');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'NAWANYINGI', 'NAWANYINGI', 'NWY-1');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'NAWANYINGI', 'MBAALA', 'MBL');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'NAWANYINGI', 'BUWOIREKU', 'BWR');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'NAWANYINGI', 'LUGOBANGO', 'LGB');
CALL InsertVillageIfNotExists('Iganga', 'NAWANYINGI', 'NAWANYINGI', 'WALUGOMA', 'WLG');

-- CENTRAL DIVISION Villages
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NAKAVULE WARD', 'LUBAALE', 'LBL');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NAKAVULE WARD', 'NAKAVULE ''A''', 'NKV-A');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NAKAVULE WARD', 'NAKAVULE ''B''', 'NKV-B');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NAKAVULE WARD', 'NAKAVULE MAIN', 'NKV-M');

CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NABIDONGHA PRISON WARD', 'BULIGO SOUTH A', 'BLG-A');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NABIDONGHA PRISON WARD', 'BULIGO SOUTH', 'BLG-S');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NABIDONGHA PRISON WARD', 'NABIDONGA C', 'NBD-C');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NABIDONGHA PRISON WARD', 'NABIDONGHA PRISONS', 'NBD-P');

CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NABIDONGHA WARD', 'NABIDONGHA A', 'NBD-A');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NABIDONGHA WARD', 'NABIDONGHA B', 'NBD-B');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'NABIDONGHA WARD', 'NABIDONGHA PROPER', 'NBD-PR');

CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'WALUGOGO WARD', 'KASOKOSO CENTRAL III', 'KSK-3');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'WALUGOGO WARD', 'WALUGOGO HOUSING ESTATE', 'WLG-H');

CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'KASOKOSO WARD', 'KASOKOSO CENTRAL I', 'KSK-1');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'KASOKOSO WARD', 'KASOKOSO CENTRAL II', 'KSK-2');

CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'BULIGO WARD', 'BULIGO NORTH', 'BLG-N');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'BULIGO WARD', 'KASOKOSO NORTH', 'KSK-N');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'BULIGO WARD', 'KASOKOSO SOUTH', 'KSK-S');
CALL InsertVillageIfNotExists('Iganga', 'CENTRAL DIVISION', 'BULIGO WARD', 'KAYAGA', 'KYG');

-- NORTHERN DIVISION Villages
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'NKONO WARD', 'NKONO II', 'NKN-2');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'NKONO WARD', 'NKONO CENTRAL', 'NKN-C');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'NKONO WARD', 'SILIVE', 'SLV');

CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'IGAMBA WARD', 'BIKADHO', 'BKD');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'IGAMBA WARD', 'IGAMBA', 'IGB');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'IGAMBA WARD', 'IGAMBA SOUTH', 'IGB-S');

CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'MUTUKULA WARD', 'MUTAMBALA', 'MTM');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'MUTUKULA WARD', 'MUTUKULA', 'MTK');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'MUTUKULA WARD', 'NKONO I', 'NKN-1');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'MUTUKULA WARD', 'NKONO III', 'NKN-3');

CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'NKATU WARD', 'KILOWA', 'KLW');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'NKATU WARD', 'NKAATU MAIN', 'NKT-M');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'NKATU WARD', 'NKAATU PROPER', 'NKT-P');

CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'BUGUMBA WARD', 'BUGUMBA A', 'BGM-A');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'BUGUMBA WARD', 'BUGUMBA B', 'BGM-B');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'BUGUMBA WARD', 'BUGUMBA CENTRAL', 'BGM-C');
CALL InsertVillageIfNotExists('Iganga', 'NORTHERN DIVISION', 'BUGUMBA WARD', 'BUGUMBA UPPER', 'BGM-U');

-- =====================================================
-- JINJA DISTRICT VILLAGES
-- =====================================================

-- BUSEDDE Subcounty Villages
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'BUGOBYA', 'BULAKABYA', 'BLK');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'BUGOBYA', 'BUGAYA', 'BGY');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'BUGOBYA', 'BUGOBYA', 'BGB');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'BUGOBYA', 'BULONDO', 'BLD');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'BUGOBYA', 'LUKONKO', 'LKK');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'BUGOBYA', 'NABIRAMA', 'NBR');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'BUGOBYA', 'NAKATI', 'NKT');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'BUGOBYA', 'NAMAZALA', 'NMZ');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'BUGOBYA', 'NANFUGAKI', 'NNF');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'BUGOBYA', 'NYENGA', 'NYG');

CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'ITAKAIBOLU', 'BUGONGOLO', 'BGG');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'ITAKAIBOLU', 'ITAKAIBOLU', 'ITK');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'ITAKAIBOLU', 'KALIRO/BUPUPA', 'KLR');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'ITAKAIBOLU', 'KASITA', 'KST');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'ITAKAIBOLU', 'KASOZI', 'KSZ');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'ITAKAIBOLU', 'KIGALAGALA', 'KGL');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'ITAKAIBOLU', 'MPAMBWE', 'MPM');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'ITAKAIBOLU', 'NAMATOLO', 'NMT');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'ITAKAIBOLU', 'WALUMBUGU', 'WLM');

CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'KISASI', 'BULINDA', 'BLD');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'KISASI', 'IRERA', 'IRR');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'KISASI', 'KAKUBA', 'KKB');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'KISASI', 'KATALAKABI', 'KTL');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'KISASI', 'KISASI', 'KSS');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'KISASI', 'NABULAGALA', 'NBL');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'KISASI', 'NAKABALE', 'NKB');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'KISASI', 'NAMAGANGA', 'NMG');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'KISASI', 'NAMAZINGIRI', 'NMZ');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'KISASI', 'NKALANGE', 'NKL');

CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NABITAMBALA', 'BUSIGE', 'BSG');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NABITAMBALA', 'BUWOLOMERA', 'BWL');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NABITAMBALA', 'BWIDHABWANGU', 'BWD');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NABITAMBALA', 'IRONGO', 'IRG');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NABITAMBALA', 'ITENGEZA', 'ITG');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NABITAMBALA', 'KISIRIRA', 'KSR');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NABITAMBALA', 'NABITAMBALA', 'NBT');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NABITAMBALA', 'NABIWAWULO', 'NBW');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NABITAMBALA', 'WANTUNDA', 'WNT');

CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NALINAIBI', 'BUDHOOMA', 'BDH');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NALINAIBI', 'KANAMA', 'KNM');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NALINAIBI', 'KIKO', 'KKO');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NALINAIBI', 'KINABIRYE', 'KNB');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NALINAIBI', 'LWINO', 'LWN');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NALINAIBI', 'NALINAIBI', 'NLN');
CALL InsertVillageIfNotExists('Jinja', 'BUSEDDE', 'NALINAIBI', 'NTINDA', 'NTD');

-- KAKIRA TOWN COUNCIL Villages
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KAKIRA WARD', '1.10', '110');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KAKIRA WARD', 'BANGALOW', 'BGL');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KAKIRA WARD', 'GO MOJA', 'GMJ');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KAKIRA WARD', 'LINE 40', 'L40');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KAKIRA WARD', 'MAJIMAZULI', 'MJM');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KAKIRA WARD', 'NTINKALU', 'NTK');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KAKIRA WARD', 'NYANZA QUARTER', 'NYQ');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KAKIRA WARD', 'SINGLE QUARTER', 'SQR');

CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KARONGO WARD', 'LAND/KATOOKE', 'LKT');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KARONGO WARD', 'KIRINYA', 'KRY');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KARONGO WARD', 'KOROKOTO', 'KRK');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KARONGO WARD', 'KHAT/KASABA', 'KHT');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KARONGO WARD', 'MUTAI', 'MTI');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KARONGO WARD', 'Q.6', 'Q06');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KARONGO WARD', 'TEREGO', 'TRG');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KARONGO WARD', 'WELFARE/STAFF QUARTERS', 'WFR');

CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MAWOITO WARD', 'KABEMBE A', 'KBM-A');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MAWOITO WARD', 'KABEMBE B', 'KBM-B');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MAWOITO WARD', 'KAGOGWA', 'KGG');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MAWOITO WARD', 'MAWOITO A', 'MWT-A');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MAWOITO WARD', 'MAWOITO B', 'MWT-B');

CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'WAIRAKA WARD', 'NALUBAALE', 'NLB');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'WAIRAKA WARD', 'WAIRAKA A', 'WRK-A');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'WAIRAKA WARD', 'WAIRAKA CENTRAL', 'WRK-C');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'WAIRAKA WARD', 'WAIRAKA COLLEGE', 'WRK-CG');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'WAIRAKA WARD', 'WALUMBE', 'WLM');

CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'CHICO WARD', 'BATI SAFI', 'BTS');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'CHICO WARD', 'KAGOMA GATE', 'KGG');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'CHICO WARD', 'KALANTIN', 'KLN');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'CHICO WARD', 'KYENDA', 'KYD');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'CHICO WARD', 'MATAFALI', 'MTF');

CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KABYAZA WARD', 'F.F.1', 'FF1');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KABYAZA WARD', 'G.G.3', 'GG3');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KABYAZA WARD', 'KABYAZA', 'KBZ');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KABYAZA WARD', 'LINE KUBWA', 'LKB');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KABYAZA WARD', 'LINE MABATI', 'LMB');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KABYAZA WARD', 'LINE RAILWAY', 'LRW');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KABYAZA WARD', 'LINE SAMIA', 'LSM');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KABYAZA WARD', 'LINE SECURITY', 'LSC');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'KABYAZA WARD', 'LINYA', 'LNY');

CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MWIRI WARD', 'BUSOGA COLLEGE MWIRI', 'BCM');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MWIRI WARD', 'KINYORO-KITESO', 'KKT');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MWIRI WARD', 'MWIRI PRIMARY SCHOOL', 'MPS');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MWIRI WARD', 'NAKIGO', 'NKG');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MWIRI WARD', 'NAMAZIBA', 'NMZ');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'MWIRI WARD', 'WAIRAKA B', 'WRK-B');

CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'POLOTA WARD', 'DUMBA', 'DMB');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'POLOTA WARD', 'NALONGO', 'NLG');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'POLOTA WARD', 'POLOTA B', 'PLT-B');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'POLOTA WARD', 'MARKET', 'MKT');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'POLOTA WARD', 'POLOTA A', 'PLT-A');
CALL InsertVillageIfNotExists('Jinja', 'KAKIRA TOWN COUNCIL', 'POLOTA WARD', 'SCHOOL', 'SCH');

-- BUTAGAYA Subcounty Villages
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'BUDIMA', 'BUDIMA', 'BDM');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'BUDIMA', 'BITULI', 'BTL');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'BUDIMA', 'BUWUMA EAST', 'BWM-E');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'BUDIMA', 'BUWUMA WEST A', 'BWM-WA');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'BUDIMA', 'BUWUMA WEST B', 'BWM-WB');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'BUDIMA', 'KABEMBE EAST', 'KBM-E');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'BUDIMA', 'KABEMBE WEST', 'KBM-W');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'BUDIMA', 'KIWAGAMA NORTH', 'KWG-N');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'BUDIMA', 'KIWAGAMA SOUTH', 'KWG-S');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'BUDIMA', 'NAKASEGETE', 'NKS');

CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'BUWALA A', 'BWL-A');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'BUWALA B', 'BWL-B');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'BUWALA C (ITANDA)', 'BWL-C');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'IWOLOLO B', 'IWL-B');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'IWOLOLO A', 'IWL-A');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'KISOZI A', 'KSZ-A');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'KISOZI B', 'KSZ-B');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'LUMULI KALUGU D', 'LML-D');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'LUMULI B', 'LML-B');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'LUMULI A', 'LML-A');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'LUMULI C', 'LML-C');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'NAKAKULWE A', 'NKK-A');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAKAKULWE (KISOZI)', 'NAKAKULWE B', 'NKK-B');

CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'BUBUGO BUGOBI', 'BBG-BG');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'BUBUGO BULONDO', 'BBG-BL');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'BUBUGO CENTRAL', 'BBG-C');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'BWEIGAMYE', 'BWG');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'BUSOWOOKO B', 'BSW-B');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'BUSOWOOKO CENTRAL', 'BSW-C');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'NABUKOSI BUGAISO A', 'NBK-A');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'NABUKOSI BUGAISO B', 'NBK-B');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'NABUKOSI CENTRAL', 'NBK-C');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'BULONDO A', 'BLD-A');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'BULONDO B', 'BLD-B');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'BUWOLOMERA', 'BWL');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'NAWAMPANDA', 'NAWAMPANDA CENTRAL', 'NWP-C');

CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'WANSIIMBA', 'BUTAGAYA WAIBUGA', 'BTG-W');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'WANSIIMBA', 'BUTAGAYA BWASE', 'BTG-B');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'WANSIIMBA', 'KIRANGA BUFUTA', 'KRG-BF');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'WANSIIMBA', 'KIRANGA BUSOOBA', 'KRG-BS');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'WANSIIMBA', 'KIRANGA BUSUSWA A', 'KRG-A');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'WANSIIMBA', 'KIRANGA BUSUSWA B', 'KRG-B');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'WANSIIMBA', 'KITENGESA BUGERERE', 'KTG-BG');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'WANSIIMBA', 'KITENGESA MPEMUKIRABUSA', 'KTG-MP');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'WANSIIMBA', 'WANSIMBA NAKABBALE', 'WNS-NK');
CALL InsertVillageIfNotExists('Jinja', 'BUTAGAYA', 'WANSIIMBA', 'WANSIMBA NAMAVUNDU', 'WNS-NM');

-- NAMAGERA TOWN COUNCIL Villages
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMAGERA WARD', 'KAMIRA', 'KMR');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMAGERA WARD', 'LUBIRA', 'LBR');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMAGERA WARD', 'MABIRA', 'MBR');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMAGERA WARD', 'BUGUMIRA A', 'BGM-A');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMAGERA WARD', 'BUGUMIRA B', 'BGM-B');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMAGERA WARD', 'NAMAGERA CENTRAL', 'NMG-C');

CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'LUBANI WARD', 'KIBUNDAIRE BUDHAGA', 'KBD-BG');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'LUBANI WARD', 'KIBUNDAIRE BUKWANGA', 'KBD-BK');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'LUBANI WARD', 'KIBUNDAIRE BUSEMBYA', 'KBD-BS');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'LUBANI WARD', 'LUBANI CENTRAL', 'LBN-C');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'LUBANI WARD', 'LUBANI WEST', 'LBN-W');

CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'MPUMWIRE WARD', 'MPUMWIRE CENTRAL', 'MPM-C');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'MPUMWIRE WARD', 'KIWUMO', 'KWM');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'MPUMWIRE WARD', 'KYEBANDO', 'KYB');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'MPUMWIRE WARD', 'NAWAGUMA A', 'NWG-A');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'MPUMWIRE WARD', 'NAWAGUMA B', 'NWG-B');

CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMWENDWA WARD', 'NAMWENDWA A', 'NMW-A');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMWENDWA WARD', 'NAMWENDWA B', 'NMW-B');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMWENDWA WARD', 'NAMWENDWA C', 'NMW-C');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMWENDWA WARD', 'NDIWANSI', 'NDW');
CALL InsertVillageIfNotExists('Jinja', 'NAMAGERA TOWN COUNCIL', 'NAMWENDWA WARD', 'NVUNWA', 'NVW');

-- BUWENGE Subcounty Villages
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'BUWEERA', 'BUWEERA EAST', 'BWR-E');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'BUWEERA', 'BUWEERA WEST', 'BWR-W');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'BUWEERA', 'BUWOLERO CENTRAL', 'BWL-C');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'BUWEERA', 'BUWOLERO IGOMBE', 'BWL-I');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'BUWEERA', 'BUWOLERO KIZINGA', 'BWL-K');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'BUWEERA', 'BUWOLERO MAWOLERO', 'BWL-M');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'BUWEERA', 'BUWOLERO NAKULYAKU', 'BWL-N');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'BUWEERA', 'YUUKA', 'YUK');

CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAGOMA', 'KAGOMA BUWOLOMERA', 'KGM-B');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAGOMA', 'KAGOMA CENTRAL', 'KGM-C');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAGOMA', 'KAGOMA MAKOTA', 'KGM-M');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAGOMA', 'MUTAI BUSIBE', 'MTI-BS');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAGOMA', 'MUTAI CENTRAL', 'MTI-C');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAGOMA', 'MUTAI KANYALE', 'MTI-K');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAGOMA', 'NAMALERE CENTRAL', 'NML-C');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAGOMA', 'NAMALERE KABAGANDA', 'NML-K');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAGOMA', 'NAMALERE SOUTH', 'NML-S');

CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'BUTEGANA EAST', 'BTG-E');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'BUTEGANA WEST', 'BTG-W');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'KAIIRA', 'KIR');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'MAWOITO BUBIRO EAST', 'MWT-BE');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'MAWOITO BUBIRO WEST', 'MWT-BW');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'MAWOITO BUKOLOKOTI', 'MWT-BK');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'MAWOITO BUKYEBAMBE', 'MWT-BY');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'MAWOITO BUYEMBA', 'MWT-BM');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'MAWOITO KIGAYA', 'MWT-KG');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'MUWANGI BUKOLWA', 'MWG-BK');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'MUWANGI BULAGALA', 'MWG-BL');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KAIIRA', 'MUWANGI BULUMBA', 'MWG-BM');

CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KITANABA', 'BUSIA II', 'BSA-2');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KITANABA', 'IDOOME', 'IDM');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KITANABA', 'ISIRI', 'ISR');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KITANABA', 'KITANABA', 'KTN');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'KITANABA', 'MPUNGWE', 'MPG');

CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'BUTANGALA A', 'BTG-A');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'BUTANGALA B', 'BTG-B');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'BUTANGALA C', 'BTG-C');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'BUTANGALA D', 'BTG-D');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'IGOMBE BULAGALA', 'IGB-BL');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'IGOMBE BUSIGE', 'IGB-BS');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'KALEBERA EAST', 'KLB-E');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'KALEBERA WEST', 'KLB-W');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'KYERINDA SOUTH', 'KYR-S');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'MAGAMAGA EAST', 'MGM-E');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'MAGAMAGA WEST', 'MGM-W');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'MUGULUKA WEST', 'MGK-W');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE', 'MAGAMAGA', 'MUGULUKA EAST', 'MGK-E');

-- BUYENGO TOWN COUNCIL Villages
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BULUGO WARD', 'BUDHUMBULI ZONE', 'BDH');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BULUGO WARD', 'BUSIA ZONE', 'BSA');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BULUGO WARD', 'KILANGIRA ZONE', 'KLG');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BULUGO WARD', 'SWANU ZONE', 'SWN');

CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BUTAMIRA WARD', 'HOSPITAL ZONE', 'HSP');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BUTAMIRA WARD', 'KALAMOYA ZONE', 'KLM');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BUTAMIRA WARD', 'MUHOFA ZONE', 'MHF');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BUTAMIRA WARD', 'MUKWANGA ZONE', 'MKW');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BUTAMIRA WARD', 'WAMUKOLO ZONE', 'WMK');

CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BUWABUZI WARD', 'KADIBA ZONE', 'KDB');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BUWABUZI WARD', 'KIGENYI ZONE', 'KGY');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BUWABUZI WARD', 'KYERINDA ZONE', 'KYR');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BUWABUZI WARD', 'MARKET ZONE', 'MKT');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'BUWABUZI WARD', 'MUNYEGERA ZONE', 'MNY');

CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'IZIRU WARD', 'BWASE', 'BWS');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'IZIRU WARD', 'CHURCH ZONE', 'CHR');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'IZIRU WARD', 'KABI ZONE', 'KAB');
CALL InsertVillageIfNotExists('Jinja', 'BUYENGO TOWN COUNCIL', 'IZIRU WARD', 'MUZIGITI ZONE', 'MZG');

-- BUWENGE TOWN COUNCIL Villages
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KALITUNSI WARD', 'BUDHOBYA CELL', 'BDH');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KALITUNSI WARD', 'BULAMA A', 'BLM-A');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KALITUNSI WARD', 'BULAMA B', 'BLM-B');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KALITUNSI WARD', 'BUSEGULA', 'BSG');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KALITUNSI WARD', 'KAYALWE A', 'KYL-A');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KALITUNSI WARD', 'KAYALWE B', 'KYL-B');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KALITUNSI WARD', 'LWANYAMA', 'LWY');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KALITUNSI WARD', 'NAKITOKOLO', 'NKT');

CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAGAIRE WARD', 'BUBANDA', 'BBD');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAGAIRE WARD', 'BUGONGWE', 'BGW');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAGAIRE WARD', 'BUKOLOBOZA', 'BKL');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAGAIRE WARD', 'KITUMBUGULU', 'KTM');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAGAIRE WARD', 'LWITAMAKOLI', 'LWT');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAGAIRE WARD', 'MUSISI', 'MSS');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAGAIRE WARD', 'NAMATA', 'NMT');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAGAIRE WARD', 'NAWAMBOGA', 'NWM');

CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KASALINA WARD', 'BUDAALI', 'BDL');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KASALINA WARD', 'BUWABUZI', 'BWB');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KASALINA WARD', 'BUYENGO', 'BYG');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KASALINA WARD', 'KAMIIGO', 'KMG');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KASALINA WARD', 'KANGULUMIRA', 'KGL');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KASALINA WARD', 'LWAMBOGO', 'LWM');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KASALINA WARD', 'NABIGWALI', 'NBG');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KASALINA WARD', 'NAKIVUMBI', 'NKV');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KASALINA WARD', 'NAMISOTA', 'NMS');

CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAMWANI WARD', 'BALIBONA', 'BLB');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAMWANI WARD', 'BUDHUMBULU', 'BDM');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAMWANI WARD', 'BUKASAMI', 'BKS');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAMWANI WARD', 'BUKYEYUNE', 'BKY');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAMWANI WARD', 'BUYONGO', 'BYG');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAMWANI WARD', 'MBAALE CELL', 'MBL');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAMWANI WARD', 'NABITOSI', 'NBT');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAMWANI WARD', 'NABULAGALA', 'NBL');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAMWANI WARD', 'NABUSAGWA', 'NBS');
CALL InsertVillageIfNotExists('Jinja', 'BUWENGE TOWN COUNCIL', 'KAMWANI WARD', 'NAKAJO', 'NKJ');

-- =====================================================
-- VERIFICATION AND COMPLETION
-- =====================================================

-- Count villages by district
SELECT 
    d.district_name,
    COUNT(v.id) as village_count
FROM districts d
LEFT JOIN subcounties s ON d.id = s.district_id
LEFT JOIN parishes p ON s.id = p.subcounty_id
LEFT JOIN villages v ON p.id = v.parish_id
WHERE d.district_name IN ('Luuka', 'Iganga', 'Jinja')
GROUP BY d.district_name
ORDER BY d.district_name;

-- Total villages
SELECT COUNT(*) as total_villages FROM villages;

-- Drop the procedure
DROP PROCEDURE IF EXISTS InsertVillageIfNotExists;

COMMIT;

SELECT 'All 801 villages successfully merged!' as Status, NOW() as CompletedAt;
